<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\DealBuilder\DealHistory;
use Mtc\MercuryDataModels\DealBuilder\DealMessage;
use Mtc\MercuryDataModels\DealBuilder\DealNote;
use Mtc\MercuryDataModels\DealBuilder\DealPartExchange;
use Mtc\MercuryDataModels\DealBuilder\Status;

class DealResource extends JsonResource
{
    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $this->resource->load([
            'status',
            'customer',
            'partExchange',
            'notes.author',
            'messages.user',
            'messages.customer',
            'history.author',
            'history.customer',
        ]);

        $this->resource->partExchange
            ->each(fn(DealPartExchange $vehicle) => $vehicle->preview_image = $this->partExPreviewImage($vehicle));
        $resource = $this->resource->toArray();
        $resource['assignee'] = $this->resource->assignee_id > 0
            ? $this->resource->user->name ?? __('labels.unknown_user')
            : __('labels.unassigned');
        $resource['time_since_submitted'] = $this->resource->time_since_submitted;
        $resource['time_since_updated'] = $this->resource->time_since_updated;
        $resource['status_name'] = $this->resource->status_id > 0
            ? $this->resource->status?->name ?? __('labels.unknown_status')
            : __('labels.draft');
        $resource['vehicle'] = new VehicleDetailsResource($this->resource->vehicle);
        $resource['history'] = $this->resource->history->map(fn(DealHistory $history) => $this->mapHistory($history));
        $resource['notes'] = $this->resource->notes->map(fn(DealNote $note) => $this->mapNote($note));
        $resource['messages'] = $this->resource->messages->map(fn(DealMessage $message) => $this->mapMessage($message));

        return [
            'deal' => $resource,
            'statuses' => Status::query()->select(['id', 'name'])->get(),
        ];
    }

    protected function mapHistory(DealHistory $history): array
    {
        return  [
            'id' => $history->id,
            'author' => $history->author?->name ?? __('labels.unknown_user'),
            'message' => __('deal_builder.status_changed_message', [
                'new_status' => $history->status?->name ?? __('labels.unknown_status'),
            ]),
            'timestamp' => $history->created_at->format('d/m/Y H:i'),
        ];
    }

    protected function mapNote(DealNote $note): array
    {
        return  [
            'id' => $note->id,
            'author' => $note->author?->name ?? __('labels.unknown_user'),
            'message' => $note->message,
            'timestamp' => $note->created_at->format('d/m/Y H:i'),
        ];
    }

    protected function mapMessage(DealMessage $message): array
    {
        return  [
            'id' => $message->id,
            'is_customer' => !empty($message->customer_id),
            'author_type' => $message->customer_id ? __('labels.customer') : __('labels.team'),
            'author' => $message->customer_id
                ? $message->customer?->first_name . ' ' . $message->customer?->last_name
                : $message->user?->name ?? __('labels.unknown_user'),
            'message' => $message->message,
            'timestamp' => $message->created_at->format('d/m/Y H:i'),
        ];
    }

    public function partExPreviewImage(DealPartExchange $vehicle): string
    {
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
                'customer' => Settings::get('imagin-placeholders-client-key') ?: config('services.imagin-studio.key'),
                'make' => $vehicle->make,
                'modelFamily' => $vehicle->model,
                'paintdescription' => $vehicle->colour,
                'modelYear' => $vehicle->manufacture_year,
                'aspectRatio' => '16:9',
                'zoomLevel' => 50,
            ]);
    }

}
