<?php

namespace Tests\Feature;

use App\Console\Commands\AutoTraderDataFetch;
use App\Facades\Settings;
use App\Jobs\FetchAutoTraderPerformanceData;
use Database\Seeders\Tenant\AutoTraderEnrichedDataSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Tests\TenantTestCase;
use Tests\TestCase;

class AutoTraderEnrichedDataFetchTest extends TenantTestCase
{
    private string $prefix;
    private string $prefixLive;

    protected function setUp(): void
    {
        parent::setUp();

        $this->prefix = 'https://api-sandbox.autotrader.co.uk';
        $this->prefixLive = 'https://api-sandbox.autotrader.co.uk';

        $this->seed(AutoTraderEnrichedDataSeeder::class);
        Settings::update('autotrader-advert-performance', true);
    }

    public function testSendsJob()
    {
        Queue::fake();
        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);
        Queue::assertPushed(FetchAutoTraderPerformanceData::class);
    }

    public function testRunJob()
    {
        /** @var Vehicle $vehicle */
        $vehicle = $this->createDealershipAndVehicle(uuid: "100");

        $this->mockHttpRequest();

        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);

        $atData = $vehicle->autoTraderData;

        $this->assertNotNull($atData);
        $this->assertEquals('GOOD', $atData->price_point);
        $this->assertEquals(5000, $atData->valuation);
        $this->assertEquals(123, $atData->search_results_7d);
        $this->assertEquals(23, $atData->ad_views);
    }

    /**
     * Ensures Autotrader service correctly uses the fallback stock ID when the vehicle is not found by UUID.
     */
    public function testJobUsesFallbackStockIdWhenVehicleNotFoundByUuid()
    {
        /** @var Vehicle $vehicle */
        $vehicle = $this->createDealershipAndVehicle(uuid: "abc123");

        $this->mockHttpRequest();

        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);

        $atData = $vehicle->autoTraderData;

        $this->assertNotNull($atData);
        $this->assertEquals('GOOD', $atData->price_point);
        $this->assertEquals(5000, $atData->valuation);
        $this->assertEquals(123, $atData->search_results_7d);
        $this->assertEquals(23, $atData->ad_views);
    }

    public function testBadResponse()
    {
        Dealership::factory()->create(['data' => ['auto-trader-location-id' => 100]]);

        /** @var Vehicle $vehicle */
        Vehicle::factory()->create(['uuid' => 100]);
        Http::fake([
            $this->prefix . '/stock*' => $this->badResponse(),
            '*' => Http::response(null, 400),
        ]);

        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);
    }


    private function stockResponse(): \GuzzleHttp\Promise\PromiseInterface
    {
        return Http::response([
            'results' => [
                [
                    'metadata' => [
                        'externalStockId' => 100,
                        'stockId' => 'abc123',
                    ],
                    'adverts' => [
                        'retailAdverts' => [
                            'priceIndicatorRating' => 'GOOD',
                        ]
                    ],
                    'vehicle' => [
                        'valuations' => [
                            'marketAverage' => [
                                'retail' => [
                                    'amountGBP' => 5000,
                                ]
                            ]
                        ]
                    ],
                    'responseMetrics' => [
                        'lastWeek' => [
                            'searchViews' => 123,
                            'advertViews' => 23,
                        ]
                    ]
                ],
            ],
            'totalResults' => 230
        ]);
    }

    private function badResponse(): \GuzzleHttp\Promise\PromiseInterface
    {
        return Http::response([
            'results' => [],
        ]);
    }

    private function mockHttpRequest(): void
    {
        Http::fake([
            $this->prefix . '/stock*' => $this->stockResponse(),
            $this->prefixLive . '/stock*' => $this->stockResponse(),
            '*' => Http::response(null, 400),
        ]);
    }

    private function createDealershipAndVehicle(string $uuid): Vehicle
    {
        $dealership = Dealership::factory()->create([
            'data' => ['auto-trader-location-id' => 100],
        ]);

        return Vehicle::factory()->create([
            'uuid' => $uuid,
            'dealership_id' => $dealership->id,
        ]);
    }
}
