<?php

namespace Tests\Unit;

use App\Facades\Settings;
use App\Mail\NewEnquiryMail;
use Carbon\Carbon;
use Database\Seeders\Global\CountrySeeder;
use Database\Seeders\Tenant\RapidRTCSettingSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\MercuryDataModels\Dealership;
use Tests\TestCase;
use Tests\UserForTenant;

class SendToRapidRTCTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;

    public function testSendEnquiry()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'failed_at' => null,
            'processed_at' => null,
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertSent(config('crm.new_enquiry_email'), function (NewEnquiryMail $mail) {
            return $mail->hasTo('john.doe@mtc.co.uk');
        });
    }

    public function testSkipsAlreadyProcessed()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'processed_at' => Carbon::now(),
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertNotSent(config('crm.new_enquiry_email'));
    }

    public function testSkipsTooManyAttempts()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'attempts' => 3,
            'data' => [
                'fallback-recipient' => 'john.doe@mtc.co.uk',
            ]
        ]);
        Mail::fake();
        Enquiries::makeAction('send-to-rapid-rtc')->handle($action, $enquiry);

        Mail::assertNotSent(NewEnquiryMail::class);
    }

    public function testGetFormFields()
    {
        $fields = Enquiries::makeAction('send-to-rapid-rtc')->formAdditionalDataFields();

        $this->assertIsArray($fields);
        $this->assertArrayHasKey('fallback-recipient', $fields);
    }

    public function testGetDealershipFields()
    {
        $fields = Enquiries::makeAction('send-to-rapid-rtc')->dealershipAdditionalDataFields();

        $this->assertIsArray($fields);
        $this->assertArrayHasKey('rapid-rtc-mailbox', $fields);
    }

    public function testGetErrorReason()
    {
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-to-rapid-rtc',
            'processed_at' => null,
            'failed_at' => null,
            'data' => [
                'fallback-recipient' => '',
            ]
        ]);
        Mail::fake();
        $handler = Enquiries::makeAction('send-to-rapid-rtc');
        $this->assertFalse($handler->handle($action, $enquiry));
        $this->assertIsString($handler->failureReason());
    }

    /**
     * A basic unit test example.
     *
     * @return void
     */
    public function testShowDealerHasRapidRtcField()
    {
        $this->seed(RapidRTCSettingSeeder::class);
        Settings::update('crm-rapid-rtc-enabled', true);

        $dealership = Dealership::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.dealerships.show', $dealership->id));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('dealership.data_fields'));
        $this->assertEquals('rapid-rtc-mailbox', $response->json('dealership.data_fields.0.id'));
    }

    public function testUpdateDealerChecksRapidRtcFields()
    {
        $this->seed(CountrySeeder::class);
        $this->seed(RapidRTCSettingSeeder::class);
        Settings::update('crm-rapid-rtc-enabled', true);

        $dealership = Dealership::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.dealerships.update', $dealership->id), [
                'name' => 'baz',
                'email' => 'john@example.com',
                'contact_no' => '123123',
                'address1' => 'foo',
                'address2' => 'baz',
                'city' => 'bar',
                'county' => 'foo-baz',
                'postcode' => '1004',
                'country' => 'LV',
            ]);

        $response->assertStatus(422);
        $this->assertArrayHasKey('data.rapid-rtc-mailbox', $response->json('errors'));
    }
}
