<?php

namespace App\Exports;

use App\Facades\Settings;
use App\Facades\Site;
use Mtc\MercuryDataModels\Vehicle;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class GoogleFeedExport implements FromQuery, WithHeadings, WithMapping
{
    protected $vehicles;

    public function __construct($rows = null)
    {
        $this->vehicles = $rows;
    }

    /**
     * Query method instead of collection
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership:id,name,address1,city,postcode,country',
                'make:id,name',
                'model:id,name',
                'bodyStyle:id,name'
            ])
            ->exportable();
    }

    /**
     * Map each row of data to the required format
     *
     * @param Vehicle $row
     * @return array
     */
    public function map($row): array
    {
        return [
            'store_code' => $this->getStoreCode(),
            'title' => $row->make?->name . ' ' . $row->model?->name . ' ' . $row->derivative,
            'id' => $row->id,
            'VIN' => $row->vin,
            'year' => $row->manufacture_year,
            'brand' => $row->make?->name,
            'model' => $row->model?->name,
            'trim' => $row->trim,
            'color' => $row->colour,
            'interior_color' => '',
            'mileage' => Settings::get('automotive-distance_measurement') === 'mi'
                ? $row->odometer_mi . ' miles'
                : $row->odometer_km . ' kilometres',
            'condition' => $row->is_new ? 'new' : 'used',
            'price' => $row->price . ' GBP',
            'dealership_name' => $row->dealership?->name,
            'dealership_address' => $row->dealership?->address1,
            'link' => Site::vehicleUrl($row),
            'image_link' =>  $row->mediaUses
                ->filter(fn($mediaUse) => $mediaUse->primary)
                ->first()
                ?->getUrl('large'),
            'additional_image_link' => $row->mediaUses
                ->reject(fn($mediaUse) => $mediaUse->primary)
                ->take(20)
                ->map(fn($mediaUse) => $mediaUse->getUrl('large'))
                ->implode(","),
            'vehicle_option' => '',
            'vehicle_fulfillment' => 'in_store:' . $this->getStoreCode(),
            'legal_disclaimer' => '',
            'google_product_category' => 'Vehicles & Parts > Vehicles > Motor Vehicles > Cars, Trucks & Vans',
            'link_template' => Site::vehicleUrl($row) . '?store={store_code}',
            'body_style' => $row->bodyStyle?->name
        ];
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'store_code',
            'title',
            'id',
            'VIN',
            'year',
            'brand',
            'model',
            'trim',
            'color',
            'interior_color',
            'mileage',
            'condition',
            'price',
            'dealership_name',
            'dealership_address',
            'link',
            'image_link',
            'additional_image_link',
            'vehicle_option',
            'vehicle_fulfillment',
            'legal_disclaimer',
            'google_product_category',
            'link_template',
            'body_style'
        ];
    }

    public function getAddress($row)
    {
        return json_encode([
            'addr1' => $row->dealership?->address1,
            'city' => $row->dealership?->city,
            'region' => $row->dealership?->country == 'GB' ? 'UK' : null,
            'postal_code' => $row->dealership?->postcode,
            'country' => $row->dealership?->country,
        ]);
    }

    protected function getStoreCode()
    {
        return Settings::get('sales-channels-google-store-code', null);
    }
}
