<?php

namespace App\Http\Controllers;

use App\Events\VehicleUpdated;
use App\ExportMapRepository;
use App\Exports\VehicleExport;
use App\Facades\Settings;
use App\Http\Requests\AddVehicleRequest;
use App\Http\Requests\CopyVehicleRequest;
use App\Http\Requests\ImportFileRequest;
use App\Http\Requests\SetFeaturedVehiclesRequest;
use App\Http\Requests\UpdateVehicleRequest;
use App\Http\Requests\UpdateVehicleStateRequest;
use App\Http\Requests\VehicleSearchRequest;
use App\Http\Resources\VehicleListResource;
use App\Http\Resources\VehicleViewResource;
use App\ImageSyncServiceHelper;
use App\Jobs\BulkImportJob;
use App\Models\BulkImportLog;
use App\Models\ExportMap;
use App\Services\OpenAI;
use App\VehicleRepository;
use App\VehicleSpec\Jobs\FetchVehicleSpecData;
use App\VehicleSpec\Jobs\VehicleCapIdLookup;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Maatwebsite\Excel\Facades\Excel;
use Mtc\MercuryDataModels\Services\FinanceService;
use Mtc\MercuryDataModels\Services\FinanceServiceHelper;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\VehicleLookup\VehicleLookupService;

class VehicleController extends Controller
{
    use DispatchesJobs;

    public function __construct()
    {
        $this->middleware([
            'permission:edit-vehicles',
        ]);
    }

    /**
     * Display a list of vehicles
     *
     * @return VehicleListResource
     */
    public function index(VehicleSearchRequest $request): VehicleListResource
    {
        $perPage = min((int) $request->input('per_page', 15), 200);

        $vehicles = Vehicle::query()
            ->with('primaryMediaUse.media')
            ->withViewCount()
            ->withSearchViewCount()
            ->withImageCount()
            ->withEnquiryCount()
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters', []))
            ->when($request->filled('selections'), fn($query) => $query->setSelections($request->getSelections()))
            ->when(
                $request->filled('search_term'),
                fn($query) => $query->whereFullText('search_content', $request->input('search_term'))
            )
            ->paginate($perPage);

        return new VehicleListResource($vehicles);
    }

    /**
     * Save a new vehicle.
     *
     * @param AddVehicleRequest $request
     * @param VehicleRepository $repository
     * @return VehicleViewResource
     */
    public function store(
        AddVehicleRequest $request,
        VehicleRepository $repository
    ): VehicleViewResource|Response {
        try {
            $vehicle = $repository->create($request);
            return new VehicleViewResource($vehicle->refresh());
        } catch (Exception $exception) {
            return response([
                'message' => $exception->getMessage()
            ], 422);
        }
    }

    /**
     * Show vehicle details.
     *
     * @param int $id
     * @return VehicleViewResource
     */
    public function show(Vehicle $vehicle): VehicleViewResource
    {
        return new VehicleViewResource($vehicle);
    }

    /**
     * Update vehicle details
     *
     * @param UpdateVehicleRequest $request
     * @param Vehicle $vehicle
     * @param VehicleRepository $repository
     * @return VehicleViewResource
     */
    public function update(
        UpdateVehicleRequest $request,
        Vehicle $vehicle,
        VehicleRepository $repository
    ): VehicleViewResource {
        if ($repository->update($request, $vehicle)) {
            Event::dispatch(new VehicleUpdated($vehicle));
        }

        return new VehicleViewResource($vehicle->refresh());
    }

    /**
     * Update status of a vehicle
     *
     * @param UpdateVehicleStateRequest $request
     * @param Vehicle $vehicle
     * @return VehicleViewResource
     */
    public function updateStatus(UpdateVehicleStateRequest $request, Vehicle $vehicle)
    {
        $vehicle->updateStatus($request->input('state'));
        return new VehicleViewResource($vehicle->refresh());
    }

    /**
     * Remove the specified vehicle from storage.
     *
     * @param Vehicle $vehicle
     * @return bool
     */
    public function destroy(Vehicle $vehicle): bool
    {
        return $vehicle->delete();
    }

    /**
     * Trigger Finance data fetch for vehicle
     *
     * @param Vehicle $vehicle
     * @param FinanceService $service
     * @return Response
     */
    public function getFinance(Request $request, Vehicle $vehicle, FinanceService $service): Response
    {
        if (FinanceServiceHelper::hasEnabledProvider() && $service->request($vehicle, $request->input())) {
            return response('ok');
        }
        return \response(['message' => 'Request failed, please try again later'], 422);
    }

    public function getCapId(Vehicle $vehicle): Response
    {
        if (!Settings::get('vehicle-spec-providers-cap-enabled')) {
            throw ValidationException::withMessages([
                'Cap Not available'
            ]);
        }
        if (!empty($vehicle->cap_id)) {
            throw ValidationException::withMessages([
                'Cap ID already set for vehicle'
            ]);
        }
        $this->dispatch(new VehicleCapIdLookup($vehicle));
        return response('ok');
    }

    /**
     * @param Vehicle $vehicle
     * @param OpenAI $openAI
     * @return JsonResponse
     * @throws ValidationException
     */
    public function getAIDescription(Vehicle $vehicle, OpenAI $openAI): JsonResponse
    {
        if (empty(Settings::get('open-ai-api_key'))) {
            throw ValidationException::withMessages([
                'AI Description Not Available'
            ]);
        }

        $description = $openAI->getDescription($vehicle);

        return response()->json([
            'description' => $description
        ]);
    }

    /**
     * Trigger Image fetch for vehicle
     *
     * @param Vehicle $vehicle
     * @return void
     */
    public function getImages(Vehicle $vehicle, ImageSyncServiceHelper $service): Response
    {
        $service->triggerSync($vehicle);
        return response('ok');
    }

    /**
     * Show list of featured vehicles
     *
     * @return VehicleListResource
     */
    public function getFeatured(): VehicleListResource
    {
        return new VehicleListResource(
            Vehicle::query()->where('featured', 1)->get()
        );
    }

    /**
     * Set vehicles as featured
     *
     * @param SetFeaturedVehiclesRequest $request
     * @return VehicleListResource
     */
    public function setFeatured(SetFeaturedVehiclesRequest $request): VehicleListResource
    {
        Vehicle::query()
            ->whereNotIn('id', $request->input('vehicle_ids'))
            ->update([
                'featured' => 0
            ]);

        Vehicle::query()
            ->whereIn('id', $request->input('vehicle_ids'))
            ->update([
                'featured' => 1
            ]);

        return $this->getFeatured();
    }

    public function getSpec(Vehicle $vehicle)
    {
        $this->dispatch(new FetchVehicleSpecData($vehicle));
    }

    public function getWarnings(VehicleRepository $repository): Response
    {
        return response([
            'warnings' => $repository->getVehicleWarnings(),
            'has_unmapped_taxonomies' => $repository->getHasUnmappedTaxonomies(),
        ]);
    }

    public function copy(
        CopyVehicleRequest $request,
        Vehicle $vehicle,
        VehicleRepository $repository
    ): VehicleViewResource {
        $vehicle = $repository->copy($request, $vehicle);
        return new VehicleViewResource($vehicle);
    }

    public function export(ExportMap $exportMap)
    {
        $fileName = 'vehicle_data.xlsx';
        if ($exportMap->exists) {
            return App::make(ExportMapRepository::class, ['export_map' => $exportMap])->download($fileName);
        }

        return Excel::download(new VehicleExport(), $fileName);
    }

    /**
     * Import seo data from file
     *
     * @param ImportFileRequest $request
     * @return bool[]
     */
    public function import(ImportFileRequest $request)
    {
        $tmpName = 'vehicle-import-' . Carbon::now()->format('Y_m_d_H_i_s') . '.'
            . $request->file('file')->getClientOriginalExtension();
        $path_to_file = 'vehicle-import/' . $tmpName;

        Storage::disk('file-storage')->put($path_to_file, $request->file('file')->getContent());
        BulkImportLog::query()
            ->create([
                'user_id' => Auth::id(),
                'object_type' => 'vehicle',
                'import_type' => $request->input('type'),
                'automations' => $request->input('automations', []),
                'filename' => $path_to_file,
            ]);
        $this->dispatch(
            (new BulkImportJob('vehicle', $path_to_file, 'edit-vehicles', $request->input('type') ?? ''))
                ->setAdditionalTasks($request->input('automations', []))
        );
        return [
            'success' => true,
        ];
    }

    public function lookup(Request $request, VehicleLookupService $lookupService): Response
    {
        $activeDriver = $lookupService->findActiveDriver();
        $data = [
            'makes' => $request->input('with_makes') ? $this->lookupMakes($activeDriver) : [],
            'models' => $request->input('with_models') ? $this->lookupModels($request, $activeDriver) : [],
            'values' => []
        ];
        try {

            $data['has_lookup'] = $lookupService->hasActiveDriver();
            $make = VehicleMake::query()->find($request->input('make_id'));
            $model = VehicleModel::query()->find($request->input('model_id'));
            $data['values'] = match ($request->input('data_type')) {
                'make' => $this->lookupMakes($activeDriver),
                'model' => $this->lookupModels($request, $activeDriver),
                'derivative' => $model ? $lookupService->findDerivatives($make, $model) : [],
                'variant' => $model ? $lookupService->findVariants($make, $model, $request->input('derivative') ?? '') : [],
                default => [],
            };
            $data['variant_type'] = $lookupService->findActiveDriver();
        } catch (\Exception $exception) {
            $data['error'] = $exception->getMessage();
        }

        return \response($data);
    }

    private function lookupMakes(?string $activeDriver): Collection
    {
        return VehicleMake::query()
            ->when($activeDriver == 'auto-trader', fn($query) => $query->whereNotNull('autotrader_id'))
            ->when($activeDriver == 'cap', fn($query) => $query->whereNotNull('cap_id'))
            ->select(['name', 'id'])
            ->get()
            ->map(fn($entry) => $entry->setHidden(['filterIndex']));
    }

    private function lookupModels(Request $request, ?string $activeDriver): Collection
    {
        return VehicleModel::query()
            ->when($activeDriver == 'auto-trader', fn($query) => $query->whereNotNull('autotrader_id'))
            ->when(
                $activeDriver == 'cap',
                fn($query) => $query->where(fn($capQuery) => $capQuery->whereNotNull('lcv_cap_id')
                    ->orWhereNotNull('cap_id'))
            )
            ->where('make_id', $request->input('make_id'))
            ->select(['name', 'id'])
            ->get()
            ->map(fn($entry) => $entry->setHidden(['filterIndex']));
    }
}
