<?php

namespace Tests\Tenant;

use App\MediaSizeRepository;
use Illuminate\Support\Facades\App;
use Mtc\ContentManager\Models\MediaSize;
use Tests\TenantTestCase;

class MediaSizeRepositoryTest extends TenantTestCase
{
    public function testCanImport()
    {
        MediaSize::factory()->create(['label' => 'foo', 'model' => 'vehicle']);
        $this->assertTrue($this->repository()->canBeImported(['label' => 'faz', 'model' => 'vehicle']));
        $this->assertTrue($this->repository()->canBeImported(['label' => 'foo', 'model' => 'offer']));
        $this->assertTrue($this->repository()->canBeImported(['label' => 'faz', 'model' => 'offer']));
        $this->assertFalse($this->repository()->canBeImported(['label' => 'foo', 'model' => 'vehicle']));
    }

    public function testExportToRemote()
    {
        MediaSize::factory(5)->create();
        $size = MediaSize::factory()->create(['label' => 'blue', 'model' => 'vehicle', 'width' => 100, 'height' => 200]);
        $size2 = MediaSize::factory()->create(['label' => 'green', 'model' => 'offer', 'width' => 100, 'height' => 200]);

        $result = $this->repository()->exportToRemote([$size->id, $size2->id]);

        $this->assertCount(2, $result);
        $this->assertArrayHasKey('label', $result[0]);
        $this->assertArrayHasKey('model', $result[0]);
        $this->assertArrayHasKey('width', $result[0]);
        $this->assertArrayHasKey('height', $result[0]);

        $this->assertEquals('blue', $result[0]['label']);
    }

    public function testCheckImportValidity()
    {
        MediaSize::factory()->create(['label' => 'foo', 'model' => 'vehicle', 'width' => 100, 'height' => 200]);
        $entry = [
            'label' => '',
            'model' => 'vehicle',
            'width' => 100,
            'height' => 100,
        ];
        $result = $this->repository()->checkImportEntryValidity($entry, []);
        $this->assertCount(1, $result['errors']);

        $entry = [
            'label' => 'foo',
            'model' => '',
            'width' => 100,
            'height' => 100,
        ];
        $result = $this->repository()->checkImportEntryValidity($entry, []);
        $this->assertCount(1, $result['errors']);

        $entry = [
            'label' => 'foo',
            'model' => 'vehicle',
            'width' => 100,
            'height' => 100,
        ];
        $result = $this->repository()->checkImportEntryValidity($entry, []);
        $this->assertCount(1, $result['errors']);

        $entry = [
            'label' => 'foo',
            'model' => 'offer',
            'width' => 100,
            'height' => 100,
        ];
        $result = $this->repository()->checkImportEntryValidity($entry, []);
        $this->assertCount(0, $result['errors']);
    }

    public function testImportRecord()
    {
        $data = [
            'label' => 'Foo',
            'model' => 'vehicle',
            'width' => 200,
            'height' => 100
        ];
        $this->assertTrue($this->repository()->importRecord($data));
        $this->assertTrue(MediaSize::query()->where('label', 'Foo')->where('model', 'vehicle')->exists());
    }

    private function repository(): MediaSizeRepository
    {
        return App::make(MediaSizeRepository::class);
    }
}
