<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use App\Jobs\FetchImaginStudioColourData;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Mtc\MercuryDataModels\KeyloopLeaseColour;
use Mtc\MercuryDataModels\VehicleMake;
use Tests\TenantTestCase;

class ImaginStudioTest extends TenantTestCase
{
    public function testDispatchJobDisabled()
    {
        (new FetchImaginStudioColourData())->handle();

        Http::assertNothingSent();
    }

    public function testUpdateSingleColour()
    {
        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-import-colour-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-client-key',
            'type' => 'string',
            'value' => 'client-key',
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Bar',
        ]);

        KeyloopLeaseColour::query()->create([
            'make_id' => $make->id,
            'colour_code' => '123',
        ]);

        Http::fake([
            "https://cdn.imagin.studio/*" => Http::response($this->getColourResponse(), 200),
        ]);

        (new FetchImaginStudioColourData())->handle();

        Http::assertSent(function (Request $request) {
            return $request->url() == 'https://cdn.imagin.studio/getPaintSwatches?customer=client-key&make=bar&paints=123';
        });

        $this->assertCount(
            1,
            KeyloopLeaseColour::query()
                ->where('make_id', $make->id)
                ->where('colour_code', '123')
                ->where('hex_code', '252424')
                ->where('gradient_hex_code', '545453')
                ->get()
        );
    }

    public function testUpdateMultipleColours()
    {
        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-import-colour-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-client-key',
            'type' => 'string',
            'value' => 'client-key',
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Bar',
        ]);

        KeyloopLeaseColour::query()->create([
            'make_id' => $make->id,
            'colour_code' => '123',
        ]);

        KeyloopLeaseColour::query()->create([
            'make_id' => $make->id,
            'colour_code' => '456',
        ]);

        Http::fake([
            "https://cdn.imagin.studio/*" => Http::response($this->getColourResponse(), 200),
        ]);

        (new FetchImaginStudioColourData())->handle();

        Http::assertSent(function (Request $request) {
            return $request->url() == 'https://cdn.imagin.studio/getPaintSwatches?customer=client-key&make=bar&paints=123%2C456';
        });

        $this->assertCount(
            1,
            KeyloopLeaseColour::query()
                ->where('make_id', $make->id)
                ->where('colour_code', '123')
                ->where('hex_code', '252424')
                ->where('gradient_hex_code', '545453')
                ->get()
        );

        $this->assertCount(
            1,
            KeyloopLeaseColour::query()
                ->where('make_id', $make->id)
                ->where('colour_code', '456')
                ->where('hex_code', '303d7a')
                ->where('gradient_hex_code', '5869ab')
                ->get()
        );
    }

    public function testUpdateMultipleMakes()
    {
        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-import-colour-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Site',
            'section' => 'Content',
            'group' => __('features.imagin_placeholders.title'),
            'name' => 'Import Colour Data',
            'config_key' => 'imagin-placeholders-client-key',
            'type' => 'string',
            'value' => 'client-key',
        ]);

        $make_bar = VehicleMake::factory()->create([
            'name' => 'Bar',
        ]);

        $make_buzz = VehicleMake::factory()->create([
            'name' => 'Buzz',
        ]);

        KeyloopLeaseColour::query()->create([
            'make_id' => $make_bar->id,
            'colour_code' => '123',
        ]);

        KeyloopLeaseColour::query()->create([
            'make_id' => $make_buzz->id,
            'colour_code' => '456',
        ]);

        Http::fake([
            "https://cdn.imagin.studio/*" => Http::response($this->getColourResponse(), 200),
        ]);

        (new FetchImaginStudioColourData())->handle();

        Http::assertSent(function (Request $request) {
            return $request->url() == 'https://cdn.imagin.studio/getPaintSwatches?customer=client-key&make=bar&paints=123';
        });

        Http::assertSent(function (Request $request) {
            return $request->url() == 'https://cdn.imagin.studio/getPaintSwatches?customer=client-key&make=buzz&paints=456';
        });

        $this->assertCount(
            1,
            KeyloopLeaseColour::query()
                ->where('make_id', $make_bar->id)
                ->where('colour_code', '123')
                ->where('hex_code', '252424')
                ->where('gradient_hex_code', '545453')
                ->get()
        );

        $this->assertCount(
            1,
            KeyloopLeaseColour::query()
                ->where('make_id', $make_buzz->id)
                ->where('colour_code', '456')
                ->where('hex_code', '303d7a')
                ->where('gradient_hex_code', '5869ab')
                ->get()
        );
    }

    private function getColourResponse()
    {
        return '{
  "make": "audi",
  "paints": {
    "123": {
      "paintId": "pspc0004",
      "paintDescription": "black",
      "nativePaintDescriptions": [
        "black",
        "brilliant-black"
      ],
      "primarySprayCan": {
        "sprayCanId": "spc0004",
        "paintType": "uni",
        "paintDescription": "Black Uni",
        "primarySprayCanLowLightRGB": "#0c0c0b",
        "primarySprayCanRGB": "#252424",
        "primarySprayCanMidLightRGB": "#252424",
        "primarySprayCanHighLightRGB": "#545453",
        "colourCluster": "black"
      }
    },
    "456": {
      "paintId": "pspc0256",
      "paintDescription": "bleu-navarra",
      "nativePaintDescriptions": [
        "bleu-navarra",
        "buitenkleur-is-blauw"
      ],
      "primarySprayCan": {
        "sprayCanId": "spc0256",
        "paintType": "unk",
        "paintDescription": "Navy Blue Metallic",
        "primarySprayCanLowLightRGB": "#0b194e",
        "primarySprayCanRGB": "#303d7a",
        "primarySprayCanMidLightRGB": "#303d7a",
        "primarySprayCanHighLightRGB": "#5869ab",
        "colourCluster": "blue"
      }
    }
  }
}';
    }
}
