<?php

namespace App\Exports;

use App\Facades\Settings;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;
use Illuminate\Support\Facades\Log;

class CarsIrelandExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::CARS_IRELAND),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::CARS_IRELAND)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                $imageBaseUrl = '';
                if (!empty($vehicle->mediaUses->first())) {
                    $mediaUrl = parse_url($vehicle->mediaUses->first()?->getUrl('1280x720'));
                    $imageBaseUrl = $mediaUrl['scheme'] . '://' . $mediaUrl['host'];
                }
                return [
                    'regnumber' => str_replace(' ', '', $vehicle->registration_number),
                    'dealerid' => $vehicle->dealership?->data['cars-ireland-dealer-id'],
                    'adid' => $vehicle->uuid,
                    'externaladid' => $vehicle->uuid,
                    'dealer' => $vehicle->dealership?->name,
                    'category' => $vehicle->type == VehicleType::LCV ? 'COMM' : 'CARS',
                    'make' => $vehicle->make?->name,
                    'model' => $vehicle->model?->name,
                    'variant' => $vehicle->derivative,
                    'colour' => $vehicle->colour,
                    'year' => $vehicle->manufacture_year,
                    'engine' => round($vehicle->engine_size_cc * 0.001, 1),
                    'enginesize' => $vehicle->engine_size_cc,
                    'mileage' => $vehicle->odometer_mi,
                    'price' => $vehicle->price,
                    'description' => $vehicle->description,
                    'location' => $vehicle->dealership?->city,
                    'fueltype' => $vehicle->fuelType?->name,
                    'bodytype' => $vehicle->bodyStyle?->name,
                    'transmissiontype' => $vehicle->transmission?->name,
                    'owners' => $vehicle->previous_owner_count,
                    'doors' => $vehicle->door_count,
                    'nctduemonth' => 0,
                    'nctdueyear' => 0,
                    'taxduemonth' => 0,
                    'taxdueyear' => 0,
                    'imagebaseurl' => $imageBaseUrl,
                    'imagesmallurl1' => $this->getImage($vehicle, $imageBaseUrl, 0, '96x64'),
                    'imagesmallurl2' => $this->getImage($vehicle, $imageBaseUrl, 1, '96x64'),
                    'imagesmallurl3' => $this->getImage($vehicle, $imageBaseUrl, 2, '96x64'),
                    'imagesmallurl4' => $this->getImage($vehicle, $imageBaseUrl, 3, '96x64'),
                    'imagethumburl1' => $this->getImage($vehicle, $imageBaseUrl, 0, '250x190'),
                    'imagethumburl2' => $this->getImage($vehicle, $imageBaseUrl, 1, '250x190'),
                    'imagethumburl3' => $this->getImage($vehicle, $imageBaseUrl, 2, '250x190'),
                    'imagethumburl4' => $this->getImage($vehicle, $imageBaseUrl, 3, '250x190'),
                    'imagelargeurl1' => $this->getImage($vehicle, $imageBaseUrl, 0, '900x600'),
                    'imagelargeurl2' => $this->getImage($vehicle, $imageBaseUrl, 1, '900x600'),
                    'imagelargeurl3' => $this->getImage($vehicle, $imageBaseUrl, 2, '900x600'),
                    'imagelargeurl4' => $this->getImage($vehicle, $imageBaseUrl, 3, '900x600'),
                    'youtubevideo' => $vehicle->main_video_url,
                    'mileagekms' => $vehicle->odometer_km,
                    'seats' => $vehicle->seats,
                    'annualroadtax' => 0,
                    'co2' => $vehicle->co2,
                    'aaapproved' => 'Yes',
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'REGNUMBER',
            'DEALERID',
            'ADID',
            'EXTERNALADID',
            'DEALER',
            'CATEGORY',
            'MAKE',
            'MODEL',
            'VARIANT',
            'COLOUR',
            'YEAR',
            'ENGINE',
            'ENGINECC',
            'MILEAGE',
            'PRICE',
            'DESCRIPTION',
            'LOCATION',
            'FUELTYPE',
            'BODYTYPE',
            'TRANSMISSIONTYPE',
            'OWNERS',
            'DOORS',
            'NCTDUEMONTH',
            'NCTDUEYEAR',
            'TAXDUEMONTH',
            'TAXDUEYEAR',
            'IMAGEBASEURL',
            'Image small URL 1',
            'Image small URL 2',
            'Image small URL 3',
            'Image small URL 4',
            'Image thumb URL 1',
            'Image thumb URL 2',
            'Image thumb URL 3',
            'Image thumb URL 4',
            'Image large URL 1',
            'Image large URL 2',
            'Image large URL 3',
            'Image large URL 4',
            'YOUTUBEVIDEO',
            'MILEAGEKMS',
            'SEATS',
            'ANNUALROADTAX',
            'CO2',
            'AA Approved',
        ];
    }

    private function getImage($vehicle, $baseUrl, $key, $size)
    {
        return str_replace($baseUrl, '', $vehicle->mediaUses->get($key)?->getUrl($size));
    }
}
