<?php

namespace App\Exports;

use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Mtc\MercuryDataModels\Vehicle;

class SilverBulletExport implements FromCollection, WithHeadingRow
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'drivetrain',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::SILVER_BULLET),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::SILVER_BULLET)
            )
            ->get()
            ->map(fn(Vehicle $vehicle) => [
                'id' => $vehicle->uuid,
                'registration_number' => $vehicle->registration_number ?? 'null',
                'registration_date' => $vehicle->first_registration_date?->format('Y-m-d H:i:s'),
                'make' => $vehicle->make?->name ?? 'null',
                'model' => $vehicle->model?->name ?? 'null',
                'derivative' => $vehicle->derivative ?? 'null',
                'transmission_type' => $vehicle->transmission?->name ?? 'null',
                'fuel_type' => $vehicle->fuelType?->name ?? 'null',
                'engine_size' => $vehicle->engine_size_cc ?? 0,
                'dealer_id' => ($vehicle->dealership->location_stock ?? $vehicle->dealership?->name) ?? 'null',
                'odometer' => $vehicle->odometer_mi ?? 0,
                'odometer_unit' => 'mi',
                'retail_price' => $vehicle->price ?? 'null',
                'reduced_price' => 'null',
                'model_year' => $vehicle->manufacture_year,
                'colour' => $vehicle->colour,
                'specific_colour' => 'null',
                'paint_type' => 'null',
                'interior_trim_colour' => 'null',
                'interior_trim_type' => 'null',
                'body_style' => $vehicle->bodyStyle?->name ?? 'null',
                'doors' => $vehicle->door_count,
                'co2' => $vehicle->co2 ?? 'null',
                'mpg' => $vehicle->mpg,
                'cap_id' => $vehicle->cap_id,
                'cap_code' => 'null',
                'siv' => 0,
                'ex_demo' => $vehicle->data['ex_demo'] ?? 0,
                'status' => 'available',
                'previous_owners' => $vehicle->previous_owner_count,
                'full_vehicle_check' => '0',
                'date_in_stock' => $vehicle->date_in_stock,
                'purchase_date' => 'null',
                'finance_option' => 'null',
                'drive_train' => $vehicle->drivetrain?->name ?? 'null',
                'ex_vat' => $vehicle->type !== VehicleType::CAR->value ? 1 : 0,
                'export_indicator' => 'null',
                'vehicle_type' => $vehicle->type,
                'images' => $vehicle->getPreviewImage('large'),
                'vin' => $vehicle->vin,
                'condition' => $vehicle->is_new ? 'new' : 'used',
            ]);
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'id',
            'registration_number',
            'registration_date',
            'make',
            'model',
            'derivative',
            'transmission_type',
            'fuel_type',
            'engine_size',
            'dealer_id',
            'odometer',
            'odometer_unit',
            'retail_price',
            'reduced_price',
            'model_year',
            'colour',
            'specific_colour',
            'paint_type',
            'interior_trim_colour',
            'interior_trim_type',
            'body_style',
            'doors',
            'co2',
            'mpg',
            'cap_id',
            'cap_code',
            'siv',
            'ex_demo',
            'status',
            'previous_owners',
            'full_vehicle_check',
            'date_in_stock',
            'purchase_date',
            'finance_option',
            'drive_train',
            'ex_vat',
            'export_indicator',
            'vehicle_type',
            'images',
            'vin',
            'condition',
        ];
    }
}
