<?php

namespace Tests\Feature;

use App\Traits\TrackedSeeding;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\TestCase;
use Tests\UserForTenant;

class LeaseMakeModelControllerTest extends TestCase
{
    use DatabaseMigrations;
    use UserForTenant;
    use TrackedSeeding;

    protected $tenancy = true;

    public function testMakeIndex()
    {
        VehicleMake::factory(5)->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.makes.index'));

        $response->assertStatus(200);

        $this->assertCount(5, $response->json('makes.data'));
        $this->assertArrayHasKey('id', $response->json('makes.data.0'));
        $this->assertArrayHasKey('name', $response->json('makes.data.0'));
        $this->assertArrayHasKey('import_enabled', $response->json('makes.data.0'));
    }

    public function testMakeUpdate()
    {
        $importable = VehicleMake::factory()->create([
            'import_enabled' => false,
        ]);

        $not_importable = VehicleMake::factory()->create([
            'import_enabled' => false,
        ]);

        $this->assertCount(2, VehicleMake::query()->where('import_enabled', false)->get());

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.makes.update'), [
                'makes' => [
                    [
                        'make_id' => $importable->id,
                        'import_enabled' => true,
                    ],
                    [
                        'make_id' => $not_importable->id,
                        'import_enabled' => false,
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertCount(
            1,
            VehicleMake::query()
                ->where('id', $importable->id)
                ->where('import_enabled', true)
                ->get()
        );

        $this->assertCount(
            1,
            VehicleMake::query()
                ->where('id', $not_importable->id)
                ->where('import_enabled', false)
                ->get()
        );
    }

    public function testModelIndex()
    {
        $importable_make_one = VehicleMake::factory()->create([
            'import_enabled' => true,
        ]);

        $importable_make_two = VehicleMake::factory()->create([
            'import_enabled' => true,
        ]);

        VehicleMake::factory(3)->create([
            'import_enabled' => false,
        ]);

        VehicleModel::factory(2)->create([
            'make_id' => $importable_make_one->id,
        ]);

        VehicleModel::factory(3)->create([
            'make_id' => $importable_make_two->id,
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.models.index'));

        $response->assertStatus(200);

        $this->assertArrayHasKey('models', $response->json());
        $this->assertCount(5, $response->json('models.data'));
        $this->assertArrayHasKey('id', $response->json('models.data.0'));
        $this->assertArrayHasKey('name', $response->json('models.data.0'));
        $this->assertArrayHasKey('import_enabled', $response->json('models.data.0'));

        $this->assertArrayHasKey('makes', $response->json());
        $expected_makes = VehicleMake::query()->where('import_enabled', true)->count();
        $this->assertCount($expected_makes, $response->json('makes'));
    }

    public function testModelUpdate()
    {
        $importable = VehicleModel::factory()->create([
            'import_enabled' => false,
        ]);

        $not_importable = VehicleModel::factory()->create([
            'import_enabled' => false,
        ]);

        $this->assertCount(2, VehicleModel::query()->where('import_enabled', false)->get());

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.leasing.models.update'), [
                'models' => [
                    [
                        'model_id' => $importable->id,
                        'import_enabled' => true,
                    ],
                    [
                        'model_id' => $not_importable->id,
                        'import_enabled' => false,
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertCount(
            1,
            VehicleModel::query()
                ->where('id', $importable->id)
                ->where('import_enabled', true)
                ->get()
        );

        $this->assertCount(
            1,
            VehicleModel::query()
                ->where('id', $not_importable->id)
                ->where('import_enabled', false)
                ->get()
        );
    }

    public function testFilterModelsByMake()
    {
        $vehicle_make_one = VehicleMake::factory()->create();
        $vehicle_make_two = VehicleMake::factory()->create();

        VehicleModel::factory(5)->create([
            'make_id' => $vehicle_make_one->id,
        ]);

        VehicleModel::factory(3)->create([
            'make_id' => $vehicle_make_two->id,
        ]);

        $this->assertCount(8, VehicleModel::all());
        $this->assertCount(5, VehicleModel::query()->where('make_id', $vehicle_make_one->id)->get());
        $this->assertCount(3, VehicleModel::query()->where('make_id', $vehicle_make_two->id)->get());

        // no filter
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.models.index'));

        $response->assertStatus(200);
        $this->assertCount(8, $response->json('models.data'));

        // filter by make
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.leasing.models.index', $vehicle_make_one->id));

        $response->assertStatus(200);
        $this->assertCount(5, $response->json('models.data'));
    }
}
