<?php

namespace App;

class TierHelper
{
    /**
     * Check if $tier is allowed for $required_tier feature
     *
     * @param string $currentTier
     * @param string $required_tier
     * @return bool
     */
    public static function isAllowed(string $currentTier, string $required_tier): bool
    {
        return match ($required_tier) {
            Tier::LITE->value => true,
            Tier::STANDARD->value => $currentTier !== Tier::LITE->value,
            Tier::PRO->value => in_array($currentTier, [Tier::PRO->value, Tier::ENTERPRISE->value]),
            Tier::ENTERPRISE->value => $currentTier === Tier::ENTERPRISE->value,
        };
    }

    /**
     * Check if $new tier is an upgrade to $current tier
     *
     * @param string $current
     * @param string $new
     * @return bool
     */
    public static function isUpgrade(string $current, string $new): bool
    {
        return $current !== $new
            && match ($current) {
                Tier::LITE->value => true,
                Tier::STANDARD->value => $new !== Tier::LITE->value,
                Tier::PRO->value => $new === Tier::ENTERPRISE->value,
                Tier::ENTERPRISE->value => false,
            };
    }

    /**
     * Allowed feature types for a tier
     *
     * @param self $tier
     * @return array
     */
    public static function allowedFeatures(Tier $tier): array
    {
        return match ($tier) {
            Tier::LITE => [],
            Tier::STANDARD => ['plugin'],
            Tier::PRO => ['plugin', 'add-on'],
            Tier::ENTERPRISE => ['plugin', 'add-on', 'theme', 'custom'],
        };
    }

    /**
     * Disallowed feature types for a tier
     *
     * @param array $tier
     * @return array
     */
    public static function disallowedFeatures(Tier $tier): array
    {
        return match ($tier) {
            Tier::ENTERPRISE => [],
            Tier::PRO => ['theme', 'custom'],
            Tier::STANDARD => ['theme', 'custom', 'add-on'],
            Tier::LITE => ['plugin', 'add-on', 'theme', 'custom'],
        };
    }

    /**
     * Get the user limit for tier
     *
     * @param string $currentTier
     * @param string $required_tier
     * @return bool
     */
    public static function userLimit(string $currentTier): int
    {
        return match ($currentTier) {
            Tier::LITE->value => 2,
            Tier::STANDARD->value => 5,
            Tier::PRO->value, Tier::ENTERPRISE->value => 999,
        };
    }

    /**
     * Which tiers are equal or higher than the required
     *
     * @param string $tier
     * @return array
     */
    public static function tiersAbove(string $tier): array
    {
        return match ($tier) {
            Tier::LITE->value => [
                Tier::STANDARD->value,
                Tier::PRO->value,
                Tier::ENTERPRISE->value,
            ],
            Tier::STANDARD->value => [
                Tier::PRO->value,
                Tier::ENTERPRISE->value,
            ],
            Tier::PRO->value => [
                Tier::ENTERPRISE->value,
            ],
            Tier::ENTERPRISE->value => [],
        };
    }

    public static function orphanedMediaAge(string $tier): int
    {
        return match ($tier) {
            Tier::LITE->value => 3,
            Tier::STANDARD->value => 6,
            Tier::PRO->value => 12,
            Tier::ENTERPRISE->value => 24,
        };
    }
}
