<?php

namespace Tests\Tenant;

use App\Services\CarAndDrivingService;
use App\Facades\Settings;
use App\VehicleType;
use Database\Seeders\Tenant\CarAndDrivingReviewSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Artisan;
use Mtc\MercuryDataModels\Vehicle;
use Database\Seeders\Tenant\CarAndDrivingSeeder;
use Tests\TestCase;
use Tests\UserForTenant;

class CarAndDrivingImportTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;

    protected function setUp(): void
    {
        parent::setUp();
 
        $this->seed(CarAndDrivingSeeder::class);
        $this->seed(CarAndDrivingReviewSeeder::class);

        Settings::shouldReceive('get')->andReturn(null)->byDefault();

        Http::preventStrayRequests();
    }

    public function testCreateReviewsAndVideo()
    { 
         Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-enabled')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-user-id')
            ->once()
            ->andReturn('test-tenant');

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-image-sync-vehicle-types', 'none')
            ->andReturn("new");

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-image-import-behaviour')
            ->andReturn("add");

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-should-sync-videos')
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-sync-type')
            ->twice()
            ->andReturn('new');
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-should-delete-existing-images', false)->andReturn(false);

        $this->mockApiResponse(); 

        $vehicle = Vehicle::factory()->create(['cap_id' => '102373', 'is_new' => 1, 'type' => VehicleType::CAR->value]);

        Artisan::call('car-and-driving:bulk-import');

        $this->assertDatabaseHas('vehicle_reviews', [
            'external_review_id' => '12767'
        ]);

        $this->assertDatabaseHas('vehicles', [
            'cap_id' => '102373',
        ]);

        $vehicle = Vehicle::where('cap_id', '102373')->first();
        $this->assertEquals('https://ssl.caranddriving.com/f6/videos/shorts700/short_Abarth500e_2023.mp4', $vehicle->main_video_url);

         $this->assertDatabaseHas('api_notifications', [
            'provider' => 'car-and-driving',
            'processed' => true,
            'data_model' => \Mtc\MercuryDataModels\VehicleReview::class,
        ]);


        $notification = \Mtc\MercuryDataModels\ApiNotification::first();
        $summary = $notification->data['summary'];

        $this->assertGreaterThan(0, $summary['total_reviews_received']);
        $this->assertGreaterThanOrEqual(1, $summary['total_reviews_processed']);
        $this->assertGreaterThanOrEqual(1, $summary['total_cap_codes_received']);
        $this->assertGreaterThanOrEqual(1, $summary['total_cap_ids_that_exists_on_site']);
        $this->assertContains('102373', $summary['cap_ids_that_exists_on_site']);

        $this->assertGreaterThanOrEqual(1, $summary['total_images_processed']);
        $this->assertGreaterThanOrEqual(1, $summary['total_videos_processed']);
    }

    public function testImportSkipsWhenIntegrationIsNotEnabled()
    {
        $this->mockApiResponse(); 

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-enabled')
            ->once()
            ->andReturn(false);

        Artisan::call('car-and-driving:bulk-import');

        $this->assertDatabaseCount('vehicle_reviews', 0);
    }

    public function testDeletesCarAndDrivingImagesWhenSettingIsActivated()
    {
        $vehicle = Vehicle::factory()->create(['cap_id' => '102373', 'is_new' => 1, 'type' => VehicleType::CAR->value]);
        $carAndDrivingMedia = \Mtc\ContentManager\Models\Media::factory()->create(['image_provider' => 'car-and-driving']);
        $otherProviderMedia = \Mtc\ContentManager\Models\Media::factory()->create(['image_provider' => 'other-provider']);

        $vehicle->mediaUses()->create(['media_id' => $carAndDrivingMedia->id]);
        $vehicle->mediaUses()->create(['media_id' => $otherProviderMedia->id]);

        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-enabled')->andReturn(true);
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-user-id')->andReturn('test-tenant');
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-image-sync-vehicle-types', 'none')->andReturn("new");
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-image-import-behaviour')->andReturn("add");
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-should-sync-videos')->andReturn(true);
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-sync-type')->andReturn('new');
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-should-delete-existing-images', false)->andReturn(true);
        Settings::shouldReceive('get')
            ->with('app-content-car-and-driving-image-overwrite-threshold', 0)->andReturn(1);

        $this->mockApiResponse();

        Artisan::call('car-and-driving:bulk-import');

        $this->assertDatabaseMissing('media_uses', [
            'media_id' => $carAndDrivingMedia->id,
            'owner_id' => $vehicle->id,
        ]);
        $this->assertDatabaseHas('media_uses', [
            'media_id' => $otherProviderMedia->id,
            'owner_id' => $vehicle->id,
        ]);
    }

    private function mockApiResponse()
    {
        $service = new CarAndDrivingService;
        Http::fake([ 
            $service->getEndpoint() . '*' => Http::response(
                file_get_contents(base_path('tests/data/car-and-driving.xml'), true),
                200
            ),
        ]);
    }
}