<?php

namespace App\Jobs;

use App\Imports\OfferImport;
use App\Imports\VehicleImport;
use App\Imports\VehicleImportWithoutHeadingRow;
use App\Models\ImportMap;
use App\VehicleSpec\Jobs\FetchVehicleListSpecData;
use App\VehicleSpec\Jobs\FetchVehicleSpecData;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\Notifications\Facades\Notification;

class BulkImportJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public $timeout = 0;

    private array $additional_tasks = [];

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly string $type,
        private readonly string $pathToFile,
        private readonly string $permission,
        private readonly string $importType = '',
    ) {
        $this->onQueue('sync');
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Config::set('excel.transactions.handler', 'null');
        try {
            match ($this->type) {
                'vehicle' => $this->handleVehicleImport(),
                'offer' => $this->handleOfferImport(),
            };

            Notification::addNotification(
                'success',
                ucfirst($this->type) . ' import finished successfully',
                [],
                $this->permission,
                $this->source(),
            );
        } catch (\Exception $exception) {
            Notification::addNotification(
                'error',
                ucfirst($this->type) . ' import failed',
                [
                    $exception->getMessage(),
                    $this->pathToFile,
                ],
                $this->permission,
                $this->source(),
            );
            Log::error('Failed bulk import job ' . tenant('id'), [
                'file exists' => Storage::disk('file-storage')->exists($this->pathToFile),
                'filename' => $this->pathToFile,
                'message' => $exception->getMessage(),
                'trace' => $exception->getTrace(),
            ]);
        }
    }

    private function handleVehicleImport()
    {
        $import_map = false;
        if (is_numeric($this->importType)) {
            /** @var ?ImportMap $import_map */
            $import_map = ImportMap::query()->find($this->importType);
        }
        if ($import_map && $import_map->data['has_header_row'] === false) {
            Excel::import(
                (new VehicleImportWithoutHeadingRow(importType: $this->importType))
                    ->setImportMap($import_map)
                    ->setAdditionalTasks($this->additional_tasks),
                $this->pathToFile,
                'file-storage'
            );
        } else {
            $import = (new VehicleImport(importType: $this->importType))
                ->setAdditionalTasks($this->additional_tasks);
            if ($import_map) {
                $import = $import->setImportMap($import_map);
            }
            Excel::import(
                $import,
                $this->pathToFile,
                'file-storage'
            );
        }
    }

    public function setAdditionalTasks(array $tasks): self
    {
        $this->additional_tasks = $tasks;
        return $this;
    }

    private function handleOfferImport()
    {
        $import_started = date('Y-m-d H:i:s');

        // we need to move the file to the local disk
        Storage::disk('local')->put($this->pathToFile, Storage::disk('file-storage')->get($this->pathToFile));
        Excel::import(new OfferImport(), $this->pathToFile, 'local');
        Storage::disk('local')->delete($this->pathToFile);

        // unpublish offers that were not in the import
        VehicleOffer::query()
            ->where('updated_at', '<', $import_started)
            ->update([
                'published' => false
        ]);
    }

    private function source(): string
    {
        return match ($this->type) {
            'vehicle' => "/manage-content/vehicles/bulk-edit/",
            'offer' => "/manage-content/offers/bulk-edit/",
        };
    }
}
