<?php

namespace Tests\Tenant;

use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleTrim;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class VehicleTrimControllerTest extends TenantTestCase
{
    use UserForTenant;

    public function testIndex()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $trim = VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-trims.index', $trim));
        $response->assertStatus(200);
        $this->assertArrayHasKey('data', $response->json());
        $this->assertCount(1, $response->json('data'));
        $this->assertEquals('foo', $response->json('data.0.title'));
    }

    public function testShow()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $trim = VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-trims.show', $trim));
        $response->assertStatus(200);
        $this->assertEquals($model->id, $response->json('model_id'));
        $this->assertEquals($make->id, $response->json('make_id'));
        $this->assertEquals('foo', $response->json('name'));
        $this->assertArrayHasKey('make_ids', $response->json('data.options'));
        $this->assertArrayHasKey('model_ids', $response->json('data.options'));
    }

    public function testUpdate()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $trim = VehicleTrim::query()->create([
            'make_id' => 99,
            'model_id' => 11,
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.vehicle-trims.show', $trim), [
                'make_id' => $make->id,
                'model_id' => $model->id,
                'name' => 'faz',
                'bhp' => 123,
                'zero_to_sixty' => 12,
                'doors' => 4,
                'seats' => 5,
            ]);
        $response->assertStatus(200);
        $this->assertEquals($model->id, $response->json('model_id'));
        $this->assertEquals($make->id, $response->json('make_id'));
        $this->assertEquals('faz', $response->json('name'));
        $this->assertEquals(123, $response->json('bhp'));
        $this->assertEquals(12, $response->json('zero_to_sixty'));
        $this->assertEquals(4, $response->json('doors'));
        $this->assertEquals(5, $response->json('seats'));
    }

    public function testDestory()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $trim = VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.vehicle-trims.destroy', $trim));
        $response->assertStatus(200);
        $this->assertFalse(VehicleTrim::query()->where('id', $trim->id)->exists());
    }

    public function testCopy()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $trim = VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'name' => 'foo',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-trims.copy', $trim));

        $response->assertStatus(201);
        $this->assertEquals(2, VehicleTrim::query()->where(['make_id' => $make->id, 'model_id' => $model->id])->count());

    }
}
