<?php

namespace App\Master\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Facades\Storage;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Factories\MasterVehicleMakeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;
use Stancl\Tenancy\Database\Concerns\CentralConnection;

class VehicleMake extends Model
{
    use CentralConnection;
    use HasFactory;
    use OrderByName;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_makes';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'name',
        'slug',
        'logo',
        'monochrome_logo',
        'autotrader_id',
        'cap_id',
        'lcv_cap_id',
        'primary_colour',
        'secondary_colour',
        'keyloop_id',
        'cars',
        'lcvs',
        'motorcycles',
    ];

    protected $appends = [
        'image',
    ];

    protected static function boot()
    {
        parent::boot();

        self::saving(function (self $make) {
            $make->ensureSlug();
        });
    }

    /**
     * Relationship with models of the make
     *
     * @return HasMany
     */
    public function models(): HasMany
    {
        return $this->hasMany(VehicleModel::class, 'make_id');
    }

    /**
     * Model factory
     *
     * @return MasterVehicleMakeFactory
     */
    protected static function newFactory()
    {
        return MasterVehicleMakeFactory::new();
    }

    public function getImageAttribute()
    {
        if (!empty($this->attributes['logo'])) {
            return [
                'alt' => $this->name,
                'src' => Storage::disk(config('filesystems.default_media'))->url($this->attributes['logo'] ?? '')
            ];
        }
        return [];
    }

    protected function logo(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => [
                'path' => $value,
                'url' => !empty($value)
                    ? Storage::disk(config('filesystems.default_media'))->url($value ?? '')
                    : '',
            ],
            set: fn ($value) => is_array($value)
                ? $value['path'] ?? null
                : $value
        );
    }

    protected function monochromeLogo(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => [
                'path' => $value,
                'url' => !empty($value)
                    ? Storage::disk(config('filesystems.default_media'))->url($value ?? '')
                    : '',
            ],
            set: fn ($value) => is_array($value)
                ? $value['path'] ?? null
                : $value
        );
    }
}
