<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Contracts\ForwardsReservationToCrm;
use App\Contracts\ForwardsServiceBookingToCrm;
use App\Facades\Settings;
use App\Services\DealerWebLmsApi;
use Illuminate\Support\Facades\App;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\Booking;
use Mtc\VehicleReservations\Reservation;

class SendToDealerWebLms implements
    EnquiryAction,
    AddsDealershipFields,
    ForwardsReservationToCrm,
    ForwardsServiceBookingToCrm
{
    use CheckForExpiredAction;

    public function __construct(private DealerWebLmsApi $api)
    {
        //
    }

    public function enabled(): bool
    {
        return Settings::get('dealerweblms-crm-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'dealerweb-dealer-id' => [
                'type' => 'text',
                'label' => 'DealerWeb Dealer ID',
            ],
        ];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [
            'dealer-web-lms-field' => [
                'type' => 'select',
                'label' => 'Field mapped against Dealer Web LMS',
                'choices' => [
                    'dealerNo' => 'Dealer Number (Manufacturer’s dealer number)',
                    'thirdParty' => 'Third Party Name (Supplier of the lead)',
                    'leadType' => "Lead type",
                    'title' => 'Title',
                    'forename' => 'Forename',
                    'surname' => 'Surname',
                    'fullName' => 'Full Name (If supplied, split into Forename and Surname)',
                    'salutation' => 'Salutation',
                    'company' => 'Company Name',
                    'houseNameNo' => 'House Name/No.',
                    'street' => 'Street',
                    'locale' => 'Suburb/Locale/District',
                    'city' => 'City',
                    'postcode' => 'Postcode',
                    'telHome' => 'Home Phone Number',
                    'telMobile' => 'Mobile Phone Number',
                    'telWork' => 'Work Phone Number',
                    'email' => 'Email Address',
                    'franchise' => 'Franchise',
                    'model' => 'Model (The model the customer is enquiring about)',
                    'newUsed' => 'New or used',
                    'pexMake' => 'Part Exchange Vehicle Make',
                    'pexModel' => 'Part Exchange Vehicle Model',
                    'pexRegNo' => 'Part Exchange Vehicle Registration Number',
                    'pexMileage' => 'Part Exchange Vehicle Mileage',
                    'included_in_detail_content' => 'Add to additional lead details',
                    'url' => 'External URL for additional vehicle details (must start with https://)',
                ],
            ]
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }

        return $this->api->sendLead($enquiry, $action);
    }

    public function sendReservation(Reservation $reservation): bool
    {
        return $this->api->sendReservation($reservation);
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return $this->api->getResponseMessage();
    }

    /**
     * Message when failure detected
     *
     * @return string
     */
    public function failureReason(): string
    {
        return $this->api->getResponseMessage() ?? 'Unknown error';
    }

    public function sendServiceBooking(Booking $booking): bool
    {
        return $this->api->sendServiceBooking($booking);
    }
}
