<?php

namespace Tests\Tenant;

use App\Imports\AutoTraderApiToVehicleImport;
use App\ValuationAdjustmentRepository;
use Mtc\MercuryDataModels\ValuationAdjustment;
use Mtc\MercuryDataModels\ValuationAdjustmentRule;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleValuation;
use Tests\TenantTestCase;

class ValuationAdjustmentRepositoryTest extends TenantTestCase
{
    public function testImportRecord()
    {
        $repository = new ValuationAdjustmentRepository();
        $this->assertEquals(0, ValuationAdjustment::query()->count());
        $this->assertTrue($repository->importRecord([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
            'rules' => [
                [
                    'column' => 'make_id',
                    'operator' => '=',
                    'value' => 10,
                ],
            ]
        ]));

        $this->assertEquals(1, ValuationAdjustment::query()->count());
        $adjustment = ValuationAdjustment::query()->first();
        $this->assertTrue($adjustment->is_active);
        $this->assertTrue($adjustment->is_increase);
        $this->assertEquals(10, $adjustment->adjustment_amount);
        $this->assertEquals('percent', $adjustment->adjustment_type);
        $this->assertEquals(1, $adjustment->rules()->count());
        $rule = $adjustment->rules()->first();
        $this->assertEquals('make_id', $rule->column);
        $this->assertEquals('=', $rule->operator);
        $this->assertEquals(10, $rule->value);
    }

    public function testCanBeImported()
    {
        $repository = new ValuationAdjustmentRepository();
        $this->assertTrue($repository->canBeImported([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ]));

        ValuationAdjustment::query()->create([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ]);

        $this->assertFalse($repository->canBeImported([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ]));
    }

    public function testExportToRemote()
    {
        $repository = new ValuationAdjustmentRepository();
        $entry = ValuationAdjustment::query()->create([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ]);
        $result = $repository->exportToRemote([ $entry->id ]);
        $this->assertEquals(1, count($result));
        $this->assertEquals($entry->id, $result[0]['id']);
        $this->assertEquals($entry->is_active, $result[0]['is_active']);
        $this->assertEquals($entry->is_increase, $result[0]['is_increase']);
        $this->assertEquals($entry->adjustment_amount, $result[0]['adjustment_amount']);
        $this->assertEquals($entry->adjustment_type, $result[0]['adjustment_type']);
    }

    public function testCheckImportEntryValidity()
    {
        $repository = new ValuationAdjustmentRepository();
        $result = $repository->checkImportEntryValidity([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ], []);
        $this->assertEmpty($result['errors']);

        ValuationAdjustment::query()->create([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ]);

        $result = $repository->checkImportEntryValidity([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_amount' => 10,
            'adjustment_type' => 'percent',
        ], []);
        $this->assertNotEmpty($result['errors']);

    }

    public function testRegistrationYearPass()
    {
        $adjustment = ValuationAdjustment::factory()->create([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_type' => 'amount',
            'adjustment_amount' => '5',
        ]);

        ValuationAdjustmentRule::query()->create([
            'adjustment_id' => $adjustment->id,
            'column' => 'date_of_registration',
            'operator' => '>',
            'value' => '2020',
        ]);

        $valuation = VehicleValuation::factory()->create([
            'date_of_registration' => '2021-01-01'
        ]);

        $this->assertTrue($adjustment->evaluate($valuation));
    }

    public function testRegistrationYearFail()
    {
        $adjustment = ValuationAdjustment::factory()->create([
            'is_active' => true,
            'is_increase' => true,
            'adjustment_type' => 'amount',
            'adjustment_amount' => '5',
        ]);

        ValuationAdjustmentRule::query()->create([
            'adjustment_id' => $adjustment->id,
            'column' => 'date_of_registration',
            'operator' => '>',
            'value' => '2020',
        ]);

        $valuation = VehicleValuation::factory()->create([
            'date_of_registration' => '2019-01-01'
        ]);

        $this->assertFalse($adjustment->evaluate($valuation));
    }

    public function testEmptyRuleEvaluatesFalse()
    {
        $valuation = VehicleValuation::factory()->create();
        $adjustment = ValuationAdjustment::query()->create();

        $this->assertFalse($adjustment->evaluate($valuation));
    }

    public function testRuleEvaluatesTrue()
    {
        $valuation = VehicleValuation::factory()->create([
            'average_price' => 123,
        ]);
        $adjustment = ValuationAdjustment::query()->create();
        ValuationAdjustmentRule::factory()->create([
            'adjustment_id' => $adjustment->id,
            'column' => 'average_price',
            'operator' => '=',
            'value' => '123',
        ]);

        $this->assertTrue($adjustment->evaluate($valuation));
    }
}
