<?php

namespace App\Mail;

use App\Facades\Settings;
use App\Facades\Site;
use App\IntegrationRepository;
use App\Traits\GetsColourSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Address;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Models\EnquiryType;
use Mtc\MercuryDataModels\Form;
use Mtc\MercuryDataModels\GlobalContent;
use Mtc\VehicleReservations\Reservation;

class NewReservationCustomerCopyNotification extends Mailable
{
    use Queueable;
    use SerializesModels;
    use GetsColourSettings;
    use CheckThemeTemplateTrait;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct(
        public readonly Reservation $reservation,
    ) {
        $this->reservation->subject = $this->reservation->vehicle?->make?->name . ' '
            . $this->reservation->vehicle?->model?->name;
    }

    public function envelope(): Envelope
    {
        return new Envelope(
            from: new Address(
                Settings::get('mail-from-email') ?: config('mail.from.address'),
                Settings::get('mail-from-name')
            ),
            subject: __('reservations::reservations.customer-autorespond', ['title' => $this->reservation->subject]),
        );
    }

    public function content(): Content
    {
        $form = Form::query()->whereHas('type', fn ($query) => $query->where('name', 'Reservation'))->first();
        $contentText = $form?->confirmation_message;
        $footerText = '';
        $globalContent = GlobalContent::query()->where('slug', 'customer-enquiry-autoresponder')->first();


        foreach ($globalContent?->content as $content) {
            if (!empty($content['children'])) {
                foreach ($content['children'] as $childContent) {
                    if ($childContent['slug'] == 'content' && empty($contentText)) {
                        $contentText = $childContent['content'];
                    } elseif ($childContent['slug'] == 'footer') {
                        $footerText = $childContent['content'];
                    }
                }
            }
        }

        return new Content(
            view: $this->getTemplateName('reservation-confirmation'),
            with: [
                'reservation' => $this->reservation,
                'theme' => Settings::get('mail-customer-theme'),
                'site_name' => Settings::get('app-name'),
                'site_logo' => Settings::get('site-logo'),
                'phone_number' => Settings::get('app-contact-phone_number'),
                'phone_number_formatted' => $this->displayNumber(Settings::get('app-contact-phone_number', '')),
                'site_url' => Site::url(''),
                'site_logo_alternate' => Settings::get('site-alternate-logo'),
                'colours' => $this->getColourSettings(),
                'content' => $contentText,
                'footer' => $footerText,
                'socials' => app(IntegrationRepository::class)->findNonEmptyValuesForType('socials')
                    ->map(fn($link, $key) => [
                        'key' => $key,
                        'url' => $link,
                    ])->values(),
            ],
        );
    }

    protected function displayNumber(string $number): string
    {
        return match (Settings::get('app-phone-number-format')) {
            '3-3-4' => substr($number, 0, 3) . ' ' . substr($number, 3, 3) . ' ' . substr($number, 6),
            '4-3-3' => substr($number, 0, 4) . ' ' . substr($number, 4, 3) . ' ' . substr($number, 7),
            '5-3-3' => substr($number, 0, 5) . ' ' . substr($number, 5, 3) . ' ' . substr($number, 8),
            '5-6' => substr($number, 0, 5) . ' ' . substr($number, 5),
            default => $number
        };
    }
}
