<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\Mail\CompareVehiclesMail;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Mail\Mailable;
use Illuminate\Support\Facades\Mail;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Jobs\VehicleComparisonMailRequest;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleView;
use Tests\TestCase;
use Tests\UserForTenant;

class VehicleComparisonMailTest extends TestCase
{
    use DatabaseMigrations;
    use UserForTenant;

    protected $tenancy = true;

    public function testVehicleViewUpdate()
    {
        Mail::fake();

        $vehicle_one = Vehicle::factory()->create([
            'slug' => 'slug_one',
        ]);

        $vehicle_two = Vehicle::factory()->create([
            'slug' => 'slug_two',
        ]);

        VehicleView::query()->create([
            'vehicle_id' => $vehicle_one->id,
            'compare_email_requests' => 5,
            'date' => Carbon::now()->format('Y-m-d'),
        ]);

        (new VehicleComparisonMailRequest(
            'recipient@test.com',
            [
                'slug_one',
                'slug_two',
            ],
            'foo.bar.com',
        ))->handle();

        $vehicle_one_views = VehicleView::query()->where('vehicle_id', $vehicle_one->id)->first();
        $vehicle_two_views = VehicleView::query()->where('vehicle_id', $vehicle_two->id)->first();

        $this->assertEquals(6, $vehicle_one_views->compare_email_requests);
        $this->assertEquals(1, $vehicle_two_views->compare_email_requests);
    }

    public function testEmailSent()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'image',
            'config_key' => 'site-logo'
        ]);

        Mail::fake();

        (new VehicleComparisonMailRequest(
            'recipient@test.com',
            [
                'slug_one',
                'slug_two',
            ],
            'foo.bar.com',
        ))->handle();

        Mail::assertSent(CompareVehiclesMail::class, function (Mailable $mail) {
            return $mail->assertSeeInHtml('foo.bar.com')
                && $mail->assertHasSubject('Your vehicle comparison');
        });
    }

    public function testApiNotification()
    {
        Mail::fake();

        $vehicle = Vehicle::factory()->create();

        (new VehicleComparisonMailRequest(
            'recipient@test.com',
            [
                $vehicle->slug,
            ],
            'foo.bar.com',
        ))->handle();

        $this->assertCount(1, ApiNotification::all());

        $api_notification = ApiNotification::query()->first();

        $this->assertEquals('comparison_email_request', $api_notification->provider);
        $this->assertTrue($api_notification->processed);
        $this->assertEquals('foo.bar.com', $api_notification->data['url']);
        $this->assertEquals('recipient@test.com', $api_notification->data['recipient']);
        $this->assertContains($vehicle->slug, $api_notification->data['slugs']);
    }
}
