<?php

namespace App\Http\Resources;

use App\Traits\FormatAsCurrency;
use App\Traits\FranchiseData;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\OfferType;
use Mtc\MercuryDataModels\VehicleOffer;

class VehicleOfferListResource extends JsonResource
{
    use FormatAsCurrency;
    use FranchiseData;

    public function __construct($resource, private $thumbDimensions = '96x64')
    {
        parent::__construct($resource);
    }

    /**
     * Set the dimensions of thumbs
     *
     * @param string $dimensions
     * @return self
     */
    public function setThumbDimensions(string $dimensions): self
    {
        $this->thumbDimensions = $dimensions;
        return $this;
    }

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource->through(fn(VehicleOffer $offer) => $this->mapOffer($offer));

            $data = $this->resource->toArray();
            $data['soft_deletes'] = true;

            $data['view_columns'] = [
                [
                    'id' => 'slug',
                    'name' => 'Slug',
                ],
                [
                    'id' => 'make_model',
                    'name' => 'Make & Model',
                    'icon' => 'cars',
                ],
                [
                    'id' => 'price',
                    'name' => 'Price',
                    'icon' => 'tag',
                ],
                [
                    'id' => 'status',
                    'name' => 'Status',
                    'icon' => 'tag',
                ],
                [
                    'id' => 'search_views',
                    'name' => 'Search appearances',
                    'icon' => 'magnifying-glass',
                ],
                [
                    'id' => 'views',
                    'name' => 'Ad views',
                    'icon' => 'eye',
                ],
                [
                    'id' => 'image_count',
                    'name' => 'Image count',
                    'icon' => 'images',
                ],
                [
                    'id' => 'enquiry_count',
                    'name' => 'Enquiry count',
                    'icon' => 'envelope-dot',
                ],
                [
                    'id' => 'days_in_stock',
                    'name' => 'Days in stock',
                    'icon' => 'truck-clock',
                ],
                [
                    'id' => 'days_since_update',
                    'name' => 'Days since last update',
                    'icon' => 'reply-clock',
                ],
            ];
            $data['default_columns'] = [
                'slug',
                'make_model',
                'price',
            ];
            return $data;
        }

        $this->resource = $this->resource->map(fn(VehicleOffer $offer) => $this->mapOffer($offer));

        return parent::toArray($request);
    }

    /**
     * Convert vehicle to the json format used in lists
     *
     * @param VehicleOffer $offer
     * @return array
     */
    protected function mapOffer(VehicleOffer $offer)
    {
        return [
            'image' => [
                'src' => $offer->getPreviewImage($this->thumbDimensions),
                'alt' => $offer->name,
            ],
            'active' => (bool)$offer->published,
            'status' => $offer->status,
            'title' => $offer->name,
            'description' => $offer->derivative,
            'metaItems' => [
                'slug' => $offer->slug,
                'make_model' => $offer->make?->name . ' ' . $offer->model?->name,
                'price' => $this->asCurrency($offer->price ?? 0, tenant()->currency),
                'status' => $offer->published ? 'Published' : 'Draft',
                'search_views' => $offer->search_views ?? 0,
                'views' => $offer->view_count ?? 0,
                'image_count' => $offer->image_count ?? 0,
                'enquiry_count' => $offer->enquiry_count ?? 0,
                'days_in_stock' => $offer->days_in_stock,
                'days_since_update' => $offer->days_since_update,
                'hello'
            ],
            'id' => $offer->id,
        ];
    }
}
