<?php

namespace App\Http\Controllers;

use App\Filter\FilterIndex;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\CatalogOffer;
use Mtc\MercuryDataModels\ContentElement;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Form;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\GlobalContent;
use Mtc\MercuryDataModels\Label;
use Mtc\MercuryDataModels\Menu;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Property;
use Mtc\MercuryDataModels\PropertyCategory;
use Mtc\MercuryDataModels\Template;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleOffer;

class OptionListController extends Controller
{
    public function index(Request $request, string $type)
    {
        $slugs = collect();
        if (in_array($type, ['makes', 'fuel-types', 'body-styles'])) {
            // Not using relationship because of the master model morphing causing issues
            $slugs = FilterIndex::query()
                ->where('filter_type', $this->getIndexType($type))
                ->pluck('slug', 'filter_id');
        }
        return match ($type) {
            'primitives' => collect(config('pages.field_types', []))
                ->map(fn($type) => App::make($type))
                ->values(),
            'dealerships' => Dealership::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'franchises' => Franchise::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'makes' => VehicleMake::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
                ]),
            'fuel-types' => FuelType::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
                ]),
            'body-styles' => BodyStyleType::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
            ]),
            'pages' => Page::query()
                ->select(['id', 'title as name'])
                ->orderBy('name')
                ->get(),
            'menus' => Menu::query()
                ->select(['id', 'title as name'])
                ->orderBy('name')
                ->get(),
            'templates' => Template::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->active()
                ->get(),
            'elements' => ContentElement::query()
                ->select(['id', 'title as name'])
                ->when($request->filled('filter'), fn($query) => $query->where('category', $request->input('filter')))
                ->orderBy('name')
                ->active()
                ->get(),
            'global-content' => GlobalContent::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'forms' => Form::query()
                ->select(['id', 'name'])
                ->where('is_active', 1)
                ->orderBy('name')
                ->get(),
            'blog-pages' => Page::query()
                ->select(['id', 'title as name'])
                ->where('category', 'blog')
                ->orderBy('name')
                ->get(),
            'news-pages' => Page::query()
                ->select(['id', 'title as name'])
                ->where('category', 'news')
                ->orderBy('name')
                ->get(),
            'offers' => VehicleOffer::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'vehicles' => Vehicle::query()
                ->select(['id', 'title', 'slug', 'registration_number'])
                ->orderBy('title')
                ->get()
                ->map(fn (Vehicle $vehicle) => [
                    'id' => $vehicle->id,
                    'name' => collect([
                        $vehicle->title,
                        $vehicle->registration_number,
                        $vehicle->slug,
                    ])->implode(' - '),
                ]),
            'taxonomies' => PropertyCategory::query()
                ->active()
                ->select(['id', 'name', 'slug as value'])
                ->get(),
            'labels' => Label::query()
                ->select(['name', 'id'])
                ->when($request->input('filter') === 'active', fn($query) => $query->where('active', 1))
                ->get(),
            'catalog-offers' => CatalogOffer::query()
                ->select(['name', 'id'])
                ->get(),

            default => Property::query()
                ->whereHas('category', fn ($query) => $query->where('slug', $type))
                ->active()
                ->select(['name', 'slug as value'])
                ->get(),
        };
    }

    private function getIndexType(string $type): string
    {
        return match ($type) {
            'makes' => 'make',
            'fuel-types' => 'fuel_type',
            'body-styles' => 'body_type',
        };
    }
}
