<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\MaxAttemptsExceededException;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Models\MediaUse;

class ImportImagesFromUrlList implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public $timeout = 0;

    public $tries = 3;
    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(
        private readonly Collection $images,
        private readonly Model $owner,
        private readonly bool $onlyWhenNoImages = false,
        private readonly ?string $imageProvider = null,
        private readonly bool $removeOthers = false,
        private readonly bool $updateAssigned = true,
        private readonly bool $updatePrimary = true,
    ) {
        $this->onQueue('bulk-media');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): void
    {
        if ($this->shouldSkip()) {
            return;
        }
        try {
            Media::setUsesForModel($this->importMedia(), $this->owner, ['primary' => $this->updatePrimary], false);
            if ($this->removeOthers && $this->imageProvider && $this->owner) {
                MediaUse::query()->whereIn('id', $this->getImagesToRemove())->delete();
            }
        } catch (\Exception $exception) {
            Log::error('Image Bulk import failed with exception: ' . $exception->getMessage(), $exception->getTrace());
        }
    }

    public function failed(\Throwable $exception): void
    {
        if (! $exception instanceof MaxAttemptsExceededException) {
            Log::error('Image list import failed with error' . $exception->getMessage(), [
                'exception' => $exception,
                'images' => $this->images,
                'owner' => $this->owner,
                'provider' => $this->imageProvider,
            ]);
        }
    }

    /**
     * Import media records for
     * @return array
     */
    private function importMedia(): array
    {
        return $this->images
            ->map(function ($image) {
                $existing = $this->alreadyExists($image);
                if ($existing) {
                    return $this->shouldTriggerEntryUpdate($existing) ? $existing : null;
                }
                try {
                    return Media::importImageFromUrl($image, '', $this->imageProvider);
                } catch (FileNotFoundException  $exception) {
                    // File does not exist at url, do not log as error
                    return null;
                } catch (\Exception $exception) {
                    Log::warning('Failed to import image', [
                        'vehicle_id' => $this->owner->id,
                        'image_data' => $image,
                        'error' => $exception->getMessage(),
                    ]);
                    return null;
                }
            })
            ->filter()
            ->pluck('id')
            ->toArray();
    }

    private function shouldSkip(): bool
    {
        return $this->onlyWhenNoImages && $this->owner->mediaUses()->count() > 0;
    }

    private function alreadyExists($image): ?Model
    {
        if (empty($this->imageProvider)) {
            return null;
        }

        return \Mtc\MercuryDataModels\Media::query()
            ->where('image_provider', $this->imageProvider)
            ->where('source_filename', $image)
            ->first();
    }

    private function getImagesToRemove(): Collection
    {
        $image_array = $this->images->toArray();
        $media_uses = $this->owner->mediaUses()->with('media')
            ->whereHas('media', fn($query) => $query->where('image_provider', $this->imageProvider))
            ->get();

        $not_in_list = $media_uses
            ->filter(fn($mediaUse) => !in_array($mediaUse->media->source_filename, $image_array))
            ->pluck('id');

        $duplicates = $media_uses->groupBy(fn($mediaUse) => $mediaUse->media->source_filename)
            ->filter(fn(Collection $group) => $group->count() > 1)
            ->map(fn(Collection $group) => $group->first())
            ->pluck('id');

        return $not_in_list->merge($duplicates);
    }

    private function shouldTriggerEntryUpdate(\Mtc\ContentManager\Contracts\Media $existing): bool
    {
        // If flag is true we always want to update
        if ($this->updateAssigned) {
            return true;
        }

        // Otherwise we only want to assign if the record is not assigned to the model
        return $existing->uses()
            ->where('owner_type', $this->owner->getMorphClass())
            ->where('owner_id', $this->owner->id)
            ->exists() === false;
    }
}
