<?php

namespace App\Services;

use App\Crm\Config\DealerWebConfig;
use App\Facades\Settings;
use App\Facades\Site;
use DOMDocument;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\ApiNotification;
use App\Services\DealershipDetection;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;

/**
 * Class DealerWebLmsApi
 *
 * Handles communication with the DealerWeb LMS API to send lead data.
 */
class DealerWebLmsApi
{
    /**
     * API endpoint for the DealerWeb LMS.
     *
     * @var string
     */
    private string $endpoint;

    /**
     * Stores the response message.
     *
     * @var string|null
     */
    protected ?string $responseMessage = null;

    /**
     * @var Response
     */
    private \Illuminate\Http\Client\Response $response;

    /**
     * DealerWebLmsApi constructor.
     *
     * @param protected readonly DealerWebConfig $config The configuration instance.
     */
    public function __construct(
        protected readonly DealerWebConfig $config,
        private readonly DealershipDetection $dealershipDetection
    ) {
        $endpoint = Settings::get('dealerweblms-endpoint');
        if (!empty($endpoint)) {
            $this->endpoint = $endpoint;
        }
    }

    /**
     * Gets the response message from the last API call.
     *
     * @return string|null The response message or null if not set.
     */
    public function getResponseMessage(): ?string
    {
        return $this->responseMessage;
    }

    /**
     * Sends lead data to the DealerWeb LMS.
     *
     * @param EnquiryModel $enquiry The enquiry model containing lead details.
     * @param EnquiryActionModel $action The action model for the enquiry.
     * @return bool True if the lead was sent successfully, false otherwise.
     */
    public function sendLead(EnquiryModel $enquiry, EnquiryActionModel $action): bool
    {
        return $this->submitLead($this->mapEnquiry($enquiry));
    }

    /**
     * Maps enquiry data to the format required by the DealerWeb LMS.
     *
     * @param EnquiryModel $enquiry The enquiry model to map.
     * @return array The mapped enquiry data.
     */
    protected function mapEnquiry(EnquiryModel $enquiry): array
    {
        $vehicle_data = [];
        $lead_type = 'NWC';
        $autonomy_dealer_id = null;

        if ($enquiry->reason_type === 'vehicle') {
            $lead_type = 'NWC';

            $vehicleId = $enquiry->reason->id ?? null;
            $vehicle_data = $this->mapVehicle($vehicleId);

            // Set the lead type to used car if vehicle is set that way
            if ($vehicle_data['NewUsed'] === 'U') {
                $lead_type = 'USE';
            }
        }

        if ($enquiry->reason_type === 'valuation') {
            $lead_type = 'USE';

            // Get the vehicle of interest from hidden form field "Vehicle"
            $vehicleHiddenField = array_filter($enquiry->details, function ($detail) {
                return $detail['question'] === 'Vehicle' && !empty($detail['answer']);
            });

            // Add vehicle data for the vehicle of interest
            if (!empty($vehicleHiddenField)) {
                $vehicleHiddenField = reset($vehicleHiddenField);
                $vehicle_of_interest_id = $vehicleHiddenField['answer'];

                $vehicle_data = array_merge($vehicle_data, $this->mapVehicle($vehicle_of_interest_id));
            }

            // Add part exchange data to Dealerweb fields from the vehicle being valued
            $part_exchange_data = [
                'PexMake' => $enquiry->reason->make ?? null,
                'PexModel' => $enquiry->reason->model ?? null,
                'PexRegNo' => $enquiry->reason->registration_number ?? null,
                'PexMileage' => $enquiry->reason->mileage ?? null,
            ];


            if ($enquiry->reason_type === 'valuation') {
                $vehicleDetail = array_filter($enquiry->details, function ($detail) {
                    return $detail['question'] === 'Vehicle' && !empty($detail['answer']);
                });

                // Set the vehicle of interest as the dealership location
                if (!empty($vehicleDetail)) {
                    $vehicleDetail = reset($vehicleDetail);
                    $vehicle_of_interest_id = $vehicleDetail['answer'];
                    $vehicle = Vehicle::find($vehicle_of_interest_id);
                    $autonomy_dealer_id = $vehicle->dealership->id ?? $autonomy_dealer_id;
                }
            }

            $vehicle_data = array_merge($vehicle_data, $part_exchange_data);
        }

        $enquiry_details = $this->mapDetails($enquiry, $vehicle_data);

        $autonomy_dealer_id = $enquiry_details['dealer_id'] ?? $autonomy_dealer_id;

        $dealerwebDealerId = $this->getDealer($enquiry, $autonomy_dealer_id)->data['dealerweb-dealer-id']
            ?? null;

        // Send the translated Dealer Web ID through email instead of our Dealership ID
        $enquiry_details['dealer_id'] = $dealerwebDealerId;

        return array_merge([
            'Username' => $this->config->username(),
            'DealerNo' => $dealerwebDealerId,
            'ThirdParty' => $this->config->dealerNo(),
            'Password' => $this->config->password(),
            'ThirdPartyID' => $enquiry->id,
            'Version' => '7',
            'LeadType' => $lead_type,
            'EnquiryType' => $enquiry->reason_type === 'valuation' ? 'VMC' : null,
        ], $enquiry_details, $vehicle_data);
    }

    protected function getDealer(EnquiryModel $enquiry, $dealer_id = null): ?Dealership
    {
        $dealer_id = $enquiry->reason->dealership->id ?? $dealer_id;

        $postcode = $enquiry->data['postcode'] ?? null;

        $fallbackDealerId = $this->config->fallbackDealership();

        return $this->dealershipDetection->determineDealer(
            $fallbackDealerId,
            $dealer_id,
            $postcode,
        );
    }

    /**
     * Maps enquiry details to the format required by the DealerWeb LMS.
     *
     * @param EnquiryModel $enquiryModel
     * @return array The mapped details array.
     */
    protected function mapDetails(EnquiryModel $enquiryModel, $vehicle_data = []): array
    {
        $mappedDetails = [];
        $includeInDetailsContent = [];

        // Loop through each question to check for 'dealer-web-lms-field' mapping
        foreach ($enquiryModel->data['questions'] as $data_question_id => $data_question) {
            $fieldMapping = $data_question['dealer-web-lms-field'] ?? null;
            $field_value = $enquiryModel->details[$data_question_id]['answer'] ?? null;

            // Turn Booleans into readable strings
            if ($data_question['component'] === 'Boolean') {
                $field_value = ($field_value == '1' || $field_value === 1) ? 'Yes' : 'No';
            }

            if (is_array($field_value)) {
                if ($fieldMapping === 'marketing_preference') {
                    $mappedDetails['NoSMS'] = !in_array('sms', $field_value);
                    $mappedDetails['NoEmail'] = !in_array('email', $field_value);
                    $mappedDetails['NoMail'] = !in_array('mail', $field_value);
                }
            } elseif ($fieldMapping === 'include_in_details_content') {
                $label = $enquiryModel->details[$data_question_id]['question'] ?? null;
                $includeInDetailsContent[] = "{$label}: {$field_value}";
            } elseif (!empty($fieldMapping) && !empty($field_value)) {
                $mappedDetails[$fieldMapping] = $field_value;
            }
        }

        if ($enquiryModel->reason_type === 'valuation' && !empty($vehicle_data)) {
            $includeInDetailsContent[] = 'Vehicle of Interest Model: ' . $vehicle_data['Model'];
            $includeInDetailsContent[] = 'Vehicle of Interest URL: ' . $vehicle_data['URL'];
        }

        // Aggregate include_in_details_content into a single "Details" field
        if (!empty($includeInDetailsContent)) {
            $detailsContent = implode(' ', $includeInDetailsContent);
            $mappedDetails['Detail'] = "<![CDATA[{$detailsContent}]]>";
        }

        return $mappedDetails;
    }

    /**
     * Maps vehicle data to the format required by the DealerWeb LMS.
     *
     * @param int|null $vehicle_id The ID of the vehicle to map.
     * @param mixed $enquiry
     * @param bool $part_exchange Include part exchange details
     * @return array The mapped vehicle data.
     */
    protected function mapVehicle($vehicle_id): array
    {
        $vehicle_data = [];
        $vehicle = Vehicle::find($vehicle_id);

        if (!empty($vehicle)) {
            $newUsed = $vehicle->is_new ? 'N' : 'U';
            $vehicle_full_model_name = '';

            if (!empty($vehicle->make->name) && !empty($vehicle->model->name)) {
                $vehicle_full_model_name = trim($vehicle->make->name
                    . ' '
                    . $vehicle->model->name
                    . ' '
                    . $vehicle->vrm_condensed);
            }

            $vehicle_data = [
                'URL' => Site::vehicleUrl($vehicle),
                'Model' => $vehicle_full_model_name,
                'NewUsed' => $newUsed,
            ];
        }

        return $vehicle_data;
    }

    /**
     * Submits the lead data to the DealerWeb LMS via SOAP request.
     *
     * @param array $lead_data The lead data array.
     * @return bool True if processed successfully
     */
    protected function submitLead(array $lead_data): bool
    {
        // Render the SOAP XML using Blade
        $soapEnvelope = view('dealer_web_lead', ['lead_data' => $lead_data])->render();

        // HTTP Headers
        $headers = [
            'Content-Type' => 'text/xml; charset=utf-8',
            'SOAPAction' => "http://services-dealerweb.8tech.co.uk/lms/AddLead",
        ];

        if (empty($this->endpoint)) {
            $this->responseMessage = 'The endpoint has not been set.';
            return false;
        }

        // Send the SOAP request
        $this->response = Http::withHeaders($headers)
            ->withBody($soapEnvelope, 'text/xml')
            ->post($this->endpoint);

        $api_notification = [
            'provider' => 'dealer-web-lms',
            'processed' => false,
            'headers' => $headers,
            'data' => [
                'body' => $soapEnvelope,
                'request' => $lead_data,
                'result' => $this->response->body(),
                'endpoint' => $this->endpoint,
            ],
            'data_model' => DealerWebLmsApi::class,
        ];

        // Check if the response was received successfully
        if (!$this->response->successful()) {
            ApiNotification::query()->create($api_notification);
            return false;
        }

        $doc = new DOMDocument('1.0', 'utf-8');
        $soapResponse = $this->response->body();
        $doc->loadXML($soapResponse);

        // Find the 'AddLeadResult' node within the response
        $addLeadResults = $doc->getElementsByTagName("AddLeadResult");
        if ($addLeadResults->length > 0) {
            $addLeadResult = $addLeadResults->item(0);

            // Retrieve values from the AddLeadResult element
            $customerId = $addLeadResult->getElementsByTagName("CustomerID")->item(0)->nodeValue ?? null;
            $code = $addLeadResult->getElementsByTagName("ErrorCode")->item(0)->nodeValue ?? null;
            $responseMessage = $addLeadResult->getElementsByTagName("ErrorMessage")->item(0)->nodeValue ?? null;

            // Set response message and check if processing succeeded
            $this->responseMessage = $responseMessage ?? 'Unknown response';
            $isProcessed = ((int)$code === 0);
        } else {
            $this->responseMessage = 'AddLeadResult element not found in response';
        }

        // Determine if the lead was processed successfully
        $api_notification['processed'] = $isProcessed;
        $api_notification['data']['message'] = $this->responseMessage;
        $api_notification['data']['code'] = $code;
        $api_notification['data']['customerId'] = $customerId;

        // Log the response details
        ApiNotification::query()->create($api_notification);

        return $isProcessed;
    }
}
