<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\Notifications\PageVersionMarkedChangesRequested;
use App\Notifications\PageVersionMarkedToBeReviewed;
use App\PageRepository;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Mtc\ContentManager\Models\Page;
use Mtc\MercuryDataModels\PageVersion;
use Tests\TestCase;
use Tests\UserForTenant;

class PageRepositoryTest extends TestCase
{
    use DatabaseMigrations;
    use UserForTenant;

    protected $tenancy = true;


    public function testHasVersioning()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'pages-versioning-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        tenant()->tier = 'lite';
        $this->assertFalse($this->repository()->hasVersioning());

        tenant()->tier = 'pro';
        $this->assertFalse($this->repository()->hasVersioning());
        Settings::update('pages-versioning-enabled', true);
        $this->assertTrue($this->repository()->hasVersioning());
    }

    public function testMarkVersionChangesRequested()
    {
        Auth::setUser($this->getUser());
        $page = Page::factory()->create();
        $version = PageVersion::factory()->create([
            'page_id' => $page->id,
            'author_id' => $this->getUser()->id
        ]);

        Notification::fake();
        $this->repository()->markVersionAsChangesRequested($page->id, $version->id);
        Notification::assertSentTo([$this->getUser()], PageVersionMarkedChangesRequested::class);
        $version->refresh();
        $this->assertTrue($version->request_changes);
    }

    public function testMarkVersionForReview()
    {
        $this->getUser()->tenants()->attach(tenant(), ['role' => 'Administrator']);
        $page = Page::factory()->create();
        $version = PageVersion::factory()->create([
            'page_id' => $page->id,
            'author_id' => $this->getUser()->id
        ]);

        Notification::fake();
        $this->repository()->markVersionForReview($page->id, $version->id);
        Notification::assertSentTo([$this->getUser()], PageVersionMarkedToBeReviewed::class);
        $version->refresh();
        $this->assertTrue($version->pending_review);
    }

    private function repository(): PageRepository
    {
        return App::make(PageRepository::class);
    }
}
