<?php

namespace App\Http\Resources;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\ContentManager\Models\Page;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Property;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class EnquiryViewResource extends \Mtc\Crm\Http\Resources\EnquiryViewResource
{
    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->load([
            'form.questionsInclTrashed',
            'form.actions',
            'type',
            'status',
            'actions',
            'valuation',
        ]);

        $this->resource->reason_url = $this->getReasonUrl();
        $this->actionData();
        $resource = $this->resource->toArray();
        $resource['details'] = $this->enrichDetails();
        $resource['created_at'] = $this->resource->created_at->format('d/m/Y H:i');
        return $resource;
    }

    private function actionData()
    {
        $this->resource->actions
            ->each(function ($action) {
                $action->name = Str::title(str_replace('-', ' ', $action->action_name));
            });
    }

    private function enrichDetails(): Collection
    {
        return collect($this->resource->details)
            ->map(function ($answer) {
                $answer['type'] = $this->resource
                    ->form
                    ->questionsInclTrashed->where('id', $answer['id'])
                    ->first()
                    ?->type;
                $answer['value'] = match ($answer['type']) {
                    'dealership_id' =>  Dealership::query()->find($answer['answer'])?->name,
                    'vehicle_id' => Vehicle::query()->find($answer['answer'])?->title,
                    'offer_id' => VehicleOffer::query()->find($answer['answer'])?->name,
                    'new_car_id' => NewCar::query()->find($answer['answer'])?->name,
                    'page_id' => Page::query()->find($answer['answer'])?->title,
                    'file_upload' => empty($answer['answer'])
                        ? null
                        : collect($answer['answer'])->map(fn($value) => $this->getEnquiryFileUrl($value)),
                    default => null,
                };
                return $answer;
            });
    }

    private function getReasonUrl(): string
    {
        return match ($this->resource->reason_type) {
            'property' => $this->getEditUrlForProperty($this->resource->reason_id),
            'new-car' => '/manage-content/new-vehicles/edit/' . $this->resource->reason_id,
            default => '/manage-content/' . Str::plural($this->resource->reason_type)
                . '/edit/' . $this->resource->reason_id
        };
    }

    private function getEnquiryFileUrl(?string $value): string
    {
        $base_url = app()->isProduction()
            ? rtrim(config('app.url'), '/') . '/api/crm/enquiries/files/' . strip_tags($value)
            : route('enquiries.getFile', strip_tags($value));

        return $base_url . '?' .
            http_build_query([
                'tenant' => tenant()->id
            ]);
    }

    private function getEditUrlForProperty(?int $reason_id): string
    {
        $property = Property::query()
            ->with('category')
            ->where('id', $reason_id)
            ->first();

        return $property
            ? '/properties/' . $property->category?->slug . '/edit/' . $property->id
            : '';
    }
}
