<?php

namespace App\Modules\Stock;

use App\Contracts\SalesChannel;
use App\Contracts\StockProvider;
use App\Events\StockSyncFinished;
use App\Facades\Settings;
use App\Imports\AutoTraderApiToVehicleImport;
use App\Jobs\AutoTraderCSVExportJob;
use App\Services\AutoTraderApi;
use App\Services\AutoTraderHub;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Mtc\AutoTraderStock\Services\AutoTraderService;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\SalesChannelHistory;
use Mtc\MercuryDataModels\Vehicle;

class AutoTraderStock implements StockProvider, SalesChannel
{
    use DispatchesJobs;

    private Collection $dealerships;

    public function __construct(
        private readonly AutoTraderApiToVehicleImport $import,
        private readonly AutoTraderHub $export
    ) {
    }


    public function enabled(): bool
    {
        return (Settings::get('stock-auto-trader-enabled', false) ?? false)
            || (Settings::get('sales-channels-auto-trader-enabled', false) ?? false);
    }

    public function name(): string
    {
        return 'AutoTrader';
    }

    /**
     * Fields to add to dealership management
     *
     * @return array[]
     */
    public function dealershipAdditionalDataFields()
    {
        return [
            'auto-trader-location-id' => [
                'type' => 'text',
                'label' => 'Dealer ID within AutoTrader',
            ],
        ];
    }

    public function runScheduledImport(): void
    {
        $this->dealerships = $this->loadDealerships();
        $this->dealerships
            ->each(fn ($dealership, $dealership_id) => $this->syncDealershipData($dealership, $dealership_id));

        Event::dispatch(new StockSyncFinished('auto-trader'));
    }

    private function syncDealershipData($dealership, $dealership_id)
    {
        $config = config('auto_trader_stock', []);
        $config['is_production'] = config('auto_trader_stock.use_sandbox') === false;

        $service = new AutoTraderService($dealership, App::make(AutoTraderApi::class));
        $vehicles = $service->fetchVehicles();
        $this->import->setDealershipId($dealership_id);
        if (Settings::get('auto-trader-stock-remove-old-vehicles')) {
            $this->import->removeOld($vehicles);
        }

        $vehicles->each(function ($vehicle_data) use ($service) {
            if (!empty($vehicle_data['vehicle']['derivativeId'])) {
                $vehicle_data['enriched_data'] = $service->fetchAdditional($vehicle_data);
            }
            if ($this->import->exists($vehicle_data) === false) {
                $this->import->add($vehicle_data);
            }
            $this->import->update($vehicle_data);
        });
    }

    public function loadDealerships()
    {
        $this->dealerships = Dealership::all()
            ->each(function (Dealership $dealership) {
                $dealership->stock_location = $dealership->data['auto-trader-location-id'] ?? null;
            });

        return $this->dealerships
            ->keyBy(fn(Dealership $dealership) => $dealership->id)
            ->map(fn(Dealership $dealership) => $dealership->data['auto-trader-location-id'] ?? null)
            ->filter();
    }

    public function runScheduledExport(): void
    {

        if (Settings::get('sales-channels-auto-trader-method') === 'ftp') {
            $this->dispatch(new AutoTraderCSVExportJob());
        } else {
            $this->exportViaApi();
        }
    }

    private function exportViaApi(): void
    {
        try {
            $this->export->setToken(Settings::get('auto-trader-hub-token'));
            $reference = $this->export->initExport();

            $query = Vehicle::query()
                ->with([
                    'dealership',
                    'make',
                    'model',
                    'fuelType',
                    'drivetrain',
                    'transmission',
                    'bodyStyle',
                    'mediaUses.media',
                    'features',
                    'specs',
                ])
                ->exportable();

            $query->chunk(100, fn(Collection $batch) => $this->export->bulkExport($reference, $batch));
            $this->export->triggerExport($reference);

            SalesChannelHistory::store('auto-trader-sales', true, $query->count() . ' records exported');
        } catch (\Exception $exception) {
            SalesChannelHistory::store('auto-trader-sales', false, $exception->getMessage());
            Log::error('AutoTrader stock export failed with error:' . $exception->getMessage(), [
                'tenant' => tenant('id'),
            ]);
        }
    }
}
