<?php

namespace Tests\Feature;

use App\Console\Commands\MtcOldStaffCheck;
use App\Master\LoginService;
use Database\Seeders\Global\GlobalRoleSeed;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Mtc\MercuryDataModels\User;
use Tests\TestCase;

class LoginServiceTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        LoginService::clearCache();
    }

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testLoggingInWithValidUser()
    {
        Http::fake([
            "https://projects.mtcmedia.co.uk/api/staff_login.php" => Http::response([
                'name' => 'John Doe',
                'exists' => true
            ]),
            "*" => Http::response([], 400),
        ]);
        $this->seed(GlobalRoleSeed::class);

        $user = User::factory()->create(['email' => 'john.doe@mtc.co.uk']);

        (new LoginService())->check([
            'email' => 'john.doe@mtc.co.uk',
            'password' => 'password'
        ]);

        $user->refresh();

        $this->assertEquals('John Doe', $user->name);
        $this->assertTrue(password_verify('password', $user->password));

        $user2 = User::factory()->create(['email' => 'john.doe@mtcmedia.co.uk']);

        (new LoginService())->check([
            'email' => 'john.doe@mtcmedia.co.uk',
            'password' => 'password'
        ]);

        $user2->refresh();
        $this->assertEquals('John Doe', $user2->name);
        $this->assertTrue(password_verify('password', $user2->password));
    }

    public function testLoggingInWithInvalidUser()
    {
        Http::fake([
            "https://projects.mtcmedia.co.uk/api/staff_login.php" => Http::response(null),
            "*" => Http::response([], 400),
        ]);
        $this->seed(GlobalRoleSeed::class);

        $user = User::factory()->create(['email' => 'john.doe@mtc.co.uk']);

        (new LoginService())->check([
            'email' => 'john.doe@mtc.co.uk',
            'password' => 'password'
        ]);

        $user->refresh();
        $this->assertFalse(password_verify('password', $user->password));

        $user2 = User::factory()->create(['email' => 'john.doe@mtcmedia.co.uk']);

        (new LoginService())->check([
            'email' => 'john.doe@mtcmedia.co.uk',
            'password' => 'password'
        ]);

        $user2->refresh();
        $this->assertFalse(password_verify('password', $user2->password));

    }
    public function testLoggingInWithoutAnExistingAccount()
    {
        Http::fake([
            "https://projects.mtcmedia.co.uk/api/staff_login.php" => Http::response([
                'name' => 'John Doe',
                'email' => 'john.doe@mtc.co.uk',
                'exists' => true
            ]),
            "*" => Http::response([], 400),
        ]);
        $this->seed(GlobalRoleSeed::class);

        (new LoginService())->check([
            'email' => 'john.doe@mtc.co.uk',
            'password' => 'password'
        ]);

        $this->assertTrue(User::query()->where('email', 'john.doe@mtc.co.uk')->exists());
    }

    public function testRemovingOldUser()
    {
        Http::fake([
            "https://projects.mtcmedia.co.uk/api/staff/check-old" => Http::response([
                'john.doe@mtcmedia.co.uk'
            ]),
            "*" => Http::response([], 400),
        ]);

        $user = User::factory()->create(['email' => 'john.doe@mtc.co.uk']);
        $user2 = User::factory()->create(['email' => 'john.doe@mtcmedia.co.uk']);
        $this->artisan(MtcOldStaffCheck::class);

        $this->assertTrue(User::query()->where('email', $user->email)->exists());
        $this->assertFalse(User::query()->where('email', $user2->email)->exists());
    }

}
