<?php

namespace App\Exports;

use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class CarWowExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::CAR_WOW),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::CAR_WOW)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'registration' => $vehicle->registration_number,
                    'external_id' => $vehicle->uuid,
                    'make' => $vehicle->make?->name,
                    'model' => $vehicle->model?->name,
                    'derivative' => $vehicle->derivative,
                    'exterior_colour' => $vehicle->colour,
                    'vin' => $vehicle->vin,
                    'year' => $vehicle->manufacture_year,
                    'variant' => $vehicle->derivative,
                    'price' => $vehicle->price,
                    'vat_included' => $vehicle->vehicle_type === VehicleType::CAR->value ? 1 : 0,
                    'mileage' => $vehicle->odometer_mi,
                    'cap_code' => $vehicle->cap_code,
                    'cap_id' => $vehicle->cap_id,
                    'body' => $vehicle->bodyStyle?->name,
                    'transmission' => $vehicle->transmission?->name,
                    'comments' => '',
                    'stock_number' => $vehicle->uuid,
                    'gearbox' => $vehicle->transmission?->name,
                    'dealer_id' => $vehicle->dealership?->data['car-wow-dealer-id'] ?? null,
                    'dealer_name' => $vehicle->dealership?->name,
                    'dealer_street_address' => $vehicle->dealership?->address1,
                    'dealer_county' => $vehicle->dealership?->county,
                    'dealer_postcode' => $vehicle->dealership?->postcode,
                    'dealer_crm_email' => $vehicle->dealership?->email,
                    'registration_date' => $vehicle->first_registration_date,
                    'is_new' => (int)$vehicle->is_new,
                    'doors' => $vehicle->door_count,
                    'engine_size' => $vehicle->engine_size_cc,
                    'fuel_type' => $vehicle->fuelType?->name,
                    'video' => '',
                    'images' => $vehicle->mediaUses
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'registration',
            'external_id',
            'make',
            'model',
            'derivative',
            'exterior_colour',
            'vin',
            'year',
            'variant',
            'price',
            'vat_included',
            'mileage',
            'cap_code',
            'cap_id',
            'body',
            'transmission',
            'comments',
            'stock_number',
            'gearbox',
            'dealer_id',
            'dealer_name',
            'dealer_street_address',
            'dealer_county',
            'dealer_postcode',
            'dealer_crm_email',
            'registration_date',
            'is_new',
            'doors',
            'engine_size',
            'fuel_type',
            'video',
            'images',
        ];
    }
}
