<?php

namespace App\Traits;

use App\Master\Models\BodyStyleType;
use App\Master\Models\DrivetrainType;
use App\Master\Models\FuelType;
use App\Master\Models\TransmissionType;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use Illuminate\Database\Eloquent\Relations\Relation;

trait ResolvesTaxonomyMapping
{
    public function resolveTaxonomyMapping($mapables, $tenant, $map): void
    {
        tenancy()->initialize($tenant);

        foreach ($mapables as $mappable) {
            $modelClass = Relation::getMorphedModel($mappable->mappable_type);
            $model = $modelClass::find($mappable->mappable_id);

            if ($model) {
                switch ($map->taxonomy_type) {
                    case 'make':
                    case 'master-make':
                        $model->make_id = $map->taxonomy_id;
                        break;
                    case 'model':
                    case 'master-model':
                        $model->model_id = $map->taxonomy_id;
                        break;
                    case 'body_type':
                    case 'master-body_type':
                    case 'body_style':
                    case 'master-body_style':
                        $model->fill(['body_style_id' => $map->taxonomy_id]);
                        break;
                    case 'drivetrain':
                    case 'master-drivetrain':
                        $model->fill(['drivetrain_id' => $map->taxonomy_id]);
                        break;
                    case 'fuel_type':
                    case 'master-fuel_type':
                        $model->fill(['fuel_type_id' => $map->taxonomy_id]);
                        if (method_exists($model, 'fuelTypes')) {
                            $model->fuelTypes()->syncWithoutDetaching([$map->taxonomy_id]);
                        }
                        break;
                    case 'transmission':
                    case 'master-transmission':
                        $model->fill(['transmission_id' => $map->taxonomy_id]);
                        break;
                    default:
                        throw new \Exception("Unknown taxonomy type: {$map->taxonomy_type}");
                }

                $model->save();

                $mappable->delete();
            }
        }
    }

    /**
     * @param mixed $activeFilter
     * @param $taxonomies
     * @return mixed
     */
    public function getFilters(mixed $activeFilter, $taxonomies)
    {
        if (empty($activeFilter) || stripos($activeFilter, 'make') !== false) {
            $taxonomies['master-make'] = VehicleMake::query()->select(['id', 'name'])->get();
        }
        if (empty($activeFilter) || stripos($activeFilter, 'body_style') !== false) {
            $taxonomies['master-body_style'] = BodyStyleType::query()->select(['id', 'name'])->get();
        }
        if (empty($activeFilter) || stripos($activeFilter, 'fuel_type') !== false) {
            $taxonomies['master-fuel_type'] = FuelType::query()->select(['id', 'name'])->get();
        }
        if (empty($activeFilter) || stripos($activeFilter, 'transmission') !== false) {
            $taxonomies['master-transmission'] = TransmissionType::query()->select(['id', 'name'])->get();
        }
        if (empty($activeFilter) || stripos($activeFilter, 'drivetrain') !== false) {
            $taxonomies['master-drivetrain'] = DrivetrainType::query()->select(['id', 'name'])->get();
        }
        if (empty($activeFilter) || stripos($activeFilter, 'model') !== false) {
            $taxonomies['master-model'] = VehicleModel::query()
                ->with('make')
                ->get()
                ->map(fn($model) => [
                    'id' => $model->id,
                    'name' => $model->name . ' (' . ($model->make?->name ?? 'Unknown Make') . ')',
                ]);
        }
        return $taxonomies;
    }
}
