<?php

namespace Tests\Tenant;

use Illuminate\Support\Facades\Queue;
use Mtc\ContentManager\Models\Media;
use Mtc\MercuryDataModels\CatalogOffer;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class CatalogOfferControllerTest extends TenantTestCase
{
    use UserForTenant;

    public function testIndex()
    {
        CatalogOffer::factory(5)->create();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson('/api/catalog-offers');

        $response->assertStatus(200);
        $this->assertArrayHasKey('data', $response->json());
        $this->assertIsArray($response->json('data'));
        $this->assertCount(5, $response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data.0'));
        $this->assertArrayHasKey('link', $response->json('data.0'));
        $this->assertArrayHasKey('id', $response->json('data.0'));
    }

    public function testShow()
    {
        $offer = CatalogOffer::factory()->create(['name' => 'foo baz bar']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.catalog-offers.show', $offer));

        $response->assertStatus(200);
        $this->assertIsArray($response->json());
        $this->assertEquals('foo baz bar', $response->json('name'));
        $this->assertArrayHasKey('media', $response->json());
        $this->assertArrayHasKey('subtitle', $response->json());
        $this->assertArrayHasKey('link', $response->json());
        $this->assertArrayHasKey('button_text', $response->json());
        $this->assertArrayHasKey('active', $response->json());
    }
    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.catalog-offers.store'), [
                'name' => 'foo baz bar'
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo baz bar', $response->json('name'));
        $this->assertEquals(0, $response->json('active'));
    }

    public function testUpdate()
    {
        Queue::fake();
        $media = Media::factory()->create();
        $offer = CatalogOffer::factory()->create(['name' => 'foo baz bar']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.catalog-offers.update', $offer), [
                'name' => 'lorem ipsum',
                'active' => true,
                'media' => [$media->id],
                'subtitle' => 'foo faz',
                'link' => '/foo-faz',
            ]);

        $response->assertStatus(200);
        $this->assertIsArray($response->json());
        $this->assertEquals('lorem ipsum', $response->json('name'));
        $offer->refresh();
        $this->assertTrue((bool)$offer->active);
        $this->assertEquals('foo faz', $offer->subtitle);
        $this->assertEquals('/foo-faz', $offer->link);
        $this->assertEquals(1, $offer->mediaUses()->count());
    }

    public function testDestroy()
    {
        $offer = CatalogOffer::factory()->create(['name' => 'foo baz bar']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.catalog-offers.destroy', $offer));

        $response->assertStatus(200);
        $this->assertFalse(CatalogOffer::query()->where('id', $offer->id)->exists());
    }
}
