<?php

namespace App\Exports;

use App\Facades\Settings;
use Mtc\MercuryDataModels\Vehicle;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class EasiChatFeedExport implements FromQuery, WithHeadings, WithMapping
{
    protected $vehicles;

    public function collection()
    {
        return $this->vehicles ?: collect();
    }

    /**
     * Query method instead of collection
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'specs',
            ])
            ->exportable();
    }

    /**
     * Map each row of data to the required format
     *
     * @param Vehicle $row
     * @return array
     */
    public function map($row): array
    {
        return [
            'Vehicle_ID' => $row->id,
            'Vehicle_Reg' => $row->vrm_condensed,
            'Colour' => $row->colour,
            'FuelType' => $row->fuelType?->name,
            'Mileage' => Settings::get('automotive-distance_measurement') === 'mi'
                ? $row->odometer_mi
                : $row->odometer_km,
            'Bodytype' => $row->bodyStyle?->name,
            'Doors' => $row->door_count,
            'Make' => $row->make?->name,
            'Model' => $row->model?->name,
            'Variant' => $row->derivative,
            'Enginesize' => $row->engine_size_cc,
            'Price' => $row->price,
            'Transmission' => $row->transmission?->name,
            'Images' => $row->mediaUses
                ->map(fn($mediaUse) => $mediaUse->getUrl('large'))
                ->values()
                ->toArray(),
            'ServiceHistory' => '',
            'PreviousOwners' => $row->previous_owner_count,
            'Category' => strtolower($row->type) == 'lcv'
                ? 'COMM'
                : 'CARS',
            'FourWheelDrive' => '',
            'Options' => '',
            'Comments' => '',
            'New' => $row->is_new ? 'Y' : 'N',
            'Used' => $row->is_new ? 'N' : 'Y',
            'Location' => $row->dealership?->franchise?->name ?? $row->dealership?->name,
            'Origin' => '',
            'V5' => '',
            'condition' => '',
            'ExDemo' => $row->is_demo ? 'Y' : 'N',
            'FranchiseApproved' => '',
            'TradePrice' => '',
            'TradePriceExtra' => '',
            'ServiceHistoryText' => '',
            'Cap_Id' => '',
            'Attention_Grabber' => $row->attention_grabber,
            'Year' => $row->manufacture_year,
        ];
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'Vehicle_ID',
            'Vehicle_Reg',
            'Colour',
            'FuelType',
            'Mileage',
            'Bodytype',
            'Doors',
            'Make',
            'Model',
            'Variant',
            'Enginesize',
            'Price',
            'Transmission',
            'Images',
            'ServiceHistory',
            'PreviousOwners',
            'Category',
            'FourWheelDrive',
            'Options',
            'Comments',
            'New',
            'Used',
            'Location',
            'Origin',
            'V5',
            'condition',
            'ExDemo',
            'FranchiseApproved',
            'TradePrice',
            'TradePriceExtra',
            'ServiceHistoryText',
            'Cap_Id',
            'Attention_Grabber',
            'Year',
        ];
    }
}
