<?php

namespace Tenant;

use App\Modules\ServiceBooking\BookingRepository;
use App\Modules\ServiceBooking\Http\Requests\BookingCalendarRequest;
use App\Modules\ServiceBooking\Http\Requests\CreateBookingRequest;
use App\Modules\ServiceBooking\Http\Requests\UpdateBookingRequest;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Booking;
use Tests\TenantTestCase;

class BookingRepositoryTest extends TenantTestCase
{
    public function testCreateBooking()
    {
        $repository = new BookingRepository();
        $this->assertEquals(0, Booking::query()->count());

        $request = (new CreateBookingRequest())->merge([
            'uuid' => Str::random(16),
            'customer_id' => null,
            'registration_number' => 'ABC123',
            'mileage' => 12345,
            'vehicle_data' => [],
            'email' => 'foo@faz.com',
            'location_id' => 1,
            'phone_number' => '1231231122',
            'first_name' => 'peter',
            'last_name' => 'parker',
        ]);

        $repository->create($request);

        $this->assertEquals(1, Booking::query()->count());
        $booking = Booking::query()->first();

        $this->assertEquals('ABC123', $booking->registration_number);
        $this->assertEquals(12345, $booking->mileage);
        $this->assertEquals('foo@faz.com', $booking->email);
        $this->assertEquals(1, $booking->location_id);
        $this->assertEquals('1231231122', $booking->phone_number);
        $this->assertEquals('peter', $booking->first_name);
        $this->assertEquals('parker', $booking->last_name);
    }

    public function testUpdateBooking()
    {
        $repository = new BookingRepository();
        $booking = Booking::factory()->create([
            'uuid' => Str::random(16),
            'email' => 'foo@faz.com'
        ]);
        $request = (new UpdateBookingRequest())->merge(['email' => 'baz@bar.com']);
        $repository->update($booking, $request);
        $booking->refresh();
        $this->assertEquals('baz@bar.com', $booking->email);
    }

    public function testGetCalendarForMonth()
    {
        $repository = new BookingRepository();
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);

        $request = (new BookingCalendarRequest())->merge([
            'period' => 'month',
            'from' => Carbon::now()->format('Y-m-d'),

        ]);
        $calendar = $repository->getCalendar($request);
        $first_week = $calendar->first();
        $this->assertEquals(7, count($first_week));
        $first_day = $first_week->first();
        $this->assertArrayHasKey('name', $first_day);
        $this->assertArrayHasKey('weekday', $first_day);
        $this->assertArrayHasKey('week_of_year', $first_day);
        $this->assertArrayHasKey('full-date', $first_day);
        $this->assertArrayHasKey('entries', $first_day);
    }
    public function testGetCalendarForWeek()
    {
        $repository = new BookingRepository();
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);
        Booking::factory()->create(['uuid' => Str::random(16)]);

        $request = (new BookingCalendarRequest())->merge([
            'period' => 'week',
            'from' => Carbon::now()->format('Y-m-d'),

        ]);
        $calendar = $repository->getCalendar($request);
        $first_week = $calendar->first();
        $this->assertEquals(7, count($first_week));
        $first_day = $first_week->first();
        $this->assertArrayHasKey('name', $first_day);
        $this->assertArrayHasKey('weekday', $first_day);
        $this->assertArrayHasKey('week_of_year', $first_day);
        $this->assertArrayHasKey('full-date', $first_day);
        $this->assertArrayHasKey('entries', $first_day);
    }
}
