<?php

namespace Tests\Tenant;

use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\VehiclePackage;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class VehiclePackageTest extends TenantTestCase
{
    use UserForTenant;

    public function testIndexPackages()
    {
        VehiclePackage::factory(5)->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-packages.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('total'));
    }

    public function testCreatePackage()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.vehicle-packages.store'), [
                'name' => 'foo',
            ]);

        $response->assertStatus(200);
        $this->assertCount(1, VehiclePackage::query()->where('name', 'foo')->get());
    }

    public function testDeletePackage()
    {
        $package_to_delete = VehiclePackage::factory()->create();
        VehiclePackage::factory(4)->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.vehicle-packages.destroy', $package_to_delete));

        $response->assertStatus(200);
        $this->assertCount(4, VehiclePackage::query()->get());
    }

    public function testUpdatePackage()
    {
        $package = VehiclePackage::factory()->create([
            'name' => 'foo',
            'conditions' => [
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'transmission_id',
                        'operator' => '=',
                        'value' => '12345'
                    ],
                ],
            ],
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.vehicle-packages.update', $package), [
                'name' => 'bar',
                'conditions' => [
                    [
                        'value' => [
                            'type' => 'and',
                            'column' => 'transmission_id',
                            'operator' => 'in',
                            'value' => '789,123'
                        ],
                    ],
                ],
            ]);

        $response->assertStatus(200);
        $this->assertCount(1, VehiclePackage::query()->where('name', 'bar')->get());
        $this->assertEquals('789,123', $response->json('conditions.0.value.value'));

        $package->refresh();
        $this->assertEquals('bar', $package->name);
        $this->assertIsArray($package->conditions[0]['value']['value']);
        $this->assertContains('123', $package->conditions[0]['value']['value']);
        $this->assertContains('789', $package->conditions[0]['value']['value']);
    }

    public function testShowPackage()
    {
        $package = VehiclePackage::factory()->create([
            'name' => 'foo',
            'conditions' => [
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'transmission_id',
                        'operator' => '=',
                        'value' => '12345'
                    ],
                ],
            ],
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.vehicle-packages.show', $package));

        $response->assertStatus(200);
        $this->assertEquals($package->id, $response->json('data.id'));
        $this->assertEquals('foo', $response->json('data.name'));
        $this->assertArrayHasKey('make.name', $response->json('available_fields'));
        $this->assertEquals(12345, $response->json('conditions.0.value.value'));
    }

    public function testApplicablePackages()
    {
        $transmission = TransmissionType::factory()->create([
            'name' => 'manual'
        ]);

        $fuel = FuelType::factory()->create([
            'name' => 'petrol'
        ]);

        $vehicle = KeyloopLeaseVehicleVariant::factory()->create([
            'transmission_id' => $transmission->id,
            'fuel_type_id' => $fuel->id,
        ]);

        // package should be applicable to vehicle
        VehiclePackage::factory()->create([
            'price' => 123,
            'vehicle_type' => 'lease-vehicle',
            'conditions' => [
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'fuel_type_id',
                        'operator' => '=',
                        'value' => $fuel->id,
                    ],
                ],
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'transmission_id',
                        'operator' => '=',
                        'value' => $transmission->id,
                    ],
                ],
            ],
        ]);

        // package should be applicable to all vehicles
        VehiclePackage::factory()->create([
            'price' => 555,
            'vehicle_type' => 'lease-vehicle',
            'conditions' => [],
        ]);

        // package not applicable to transmission id
        VehiclePackage::factory()->create([
            'price' => 234,
            'vehicle_type' => 'lease-vehicle',
            'conditions' => [
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'fuel_type_id',
                        'operator' => '=',
                        'value' => $fuel->id,
                    ],
                ],
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'transmission_id',
                        'operator' => '=',
                        'value' => $transmission->id + 1,
                    ],
                ],
            ],
        ]);

        // package not applicable to vehicle type
        VehiclePackage::factory()->create([
            'price' => 456,
            'vehicle_type' => 'vehicle',
            'conditions' => [
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'fuel_type_id',
                        'operator' => '=',
                        'value' => $fuel->id,
                    ],
                ],
                [
                    'value' => [
                        'type' => 'and',
                        'column' => 'transmission_id',
                        'operator' => '=',
                        'value' => $transmission->id,
                    ],
                ],
            ],
        ]);

        $packages = $vehicle->getPackages();
        $prices = $packages->map(fn ($package) => $package->price);

        $this->assertCount(2, $packages);
        $this->assertContains(123, $prices);
        $this->assertContains(555, $prices);
    }
}
