<?php

namespace Tests\Feature\Commands\SyncVehicles;

use App\Events\NewVehicleImported;
use App\Events\StockSyncFinished;
use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Event;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Illuminate\Support\Facades\Http;
use App\TaxonomyMap;
use Database\Seeders\Global\CountrySeeder;
use Illuminate\Support\Facades\Config;
use Mtc\Notifications\Facades\Notification;
use Illuminate\Support\Facades\Artisan;
use Mtc\MercuryDataModels\Setting;
use Tests\TestCase;
use Tests\UserForTenant;

class AutoTraderStockImportTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    private string $prefix;

    protected $tenancy = true;

    protected function setUp(): void
    {
        parent::setUp();

        $this->prefix = 'https://api-sandbox.autotrader.co.uk';

        Config::set('queue.default', 'sync');
        Settings::shouldReceive('get')->andReturn(null)->byDefault();

        Settings::shouldReceive('get')
            ->with('stock-sync-frequency')
            ->andReturn('every-hour');

        Http::preventStrayRequests();
    }

    public function testAutoTraderStockImportForExistingVehicle()
    {
        Event::fake([StockSyncFinished::class]);

        $dealership = Dealership::factory()->create([
            'data' => ['auto-trader-location-id' => '1234'],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'vin' => 'test1 vin',
            'stock_provider' => 'autotrader',
            'was_recently_synced' => 0,
            'auto_trader_id' => 'test1'
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'master-make',
            'taxonomy_id' => 55,
            'term' => 'Ford',
            'provider' => 'autotrader',
        ]);

        $this->seed(CountrySeeder::class);

        // todo ::: this is where I am now. TO BE CONTINUED

        Setting::factory()->create([
            'config_key' => 'stock-auto-trader-enabled',
            'value' => 1
        ]);

        $this->mockAutotraderHttpResonses();
        $this->mockSettingsFacade();

        // ensures fullSync returns true
        $this->travelTo(now()->setHour(4));

        Artisan::call('sync:vehicles');

        Http::assertSent(function ($request) {
            return str_contains($request->url(), $this->prefix . '/authenticate');
        });

        Http::assertSent(function ($request) {
            return str_contains($request->url(), $this->prefix . '/stock') && str_contains($request->url(), '?');
        });

        // Assert: vehicle was updated with data from AT
        $this->assertDatabaseHas('vehicles', [
            'auto_trader_id' => 'test1',
            'derivative' => '1.0 test derivative',
            'colour' => 'Black',
            'attention_grabber' => 'attention test1',
        ]);

        Event::assertDispatched(StockSyncFinished::class, function ($event) {
            return $event->provider === 'auto-trader';
        });

        $notifications = Notification::getAll($this->getUser());
        $this->assertTrue(
            $notifications->contains(function ($notification) {
                return $notification->severity === 'info';
            }),
            'autotrader'
        );

        $this->assertDatabaseHas('vehicle_attributes', [
            'slug' => 'vat_status',
            'name' => 'VAT Status',
            'type' => 'text',
        ]);

        $this->assertDatabaseHas('vehicle_attribute_values', [
            'owner_id' => $vehicle->id,
            'slug' => 'vat_status',
            'value_text' => 'Inc VAT',
        ]);

    }

    public function testAutoTraderStockImportForNewVehicle()
    {
        Event::fake([NewVehicleImported::class]);

        Dealership::factory()->create([
            'data' => ['auto-trader-location-id' => 'abc123'],
        ]);

        TaxonomyMap::query()->create([
            'taxonomy_type' => 'master-make',
            'taxonomy_id' => 55,
            'term' => fake()->text(12),
            'provider' => 'autotrader',
        ]);

        $this->seed(CountrySeeder::class);

        Setting::factory()->create([
            'config_key' => 'stock-auto-trader-enabled',
            'value' => 1
        ]);

        $this->mockAutotraderHttpResonses();

        $this->mockSettingsFacade();

        // these settings are specific to JUST these test case
        Settings::shouldReceive('get')
            ->with('stock-auto-trader-sync-images')
            ->once()
            ->andReturn(false);
        Settings::shouldReceive('get')
            ->with('stock-auto-trader-sync-new-vehicles')
            ->twice()
            ->andReturn(true);
        Settings::shouldReceive('get')
            ->with('app-details-country')
            ->once()
            ->andReturn(null);
        Settings::shouldReceive('get')
            ->with('auto-trader-stock-remove-old-vehicles')
            ->once()
            ->andReturn(true);


        // ensures fullSync returns true
        $this->travelTo(now()->setHour(4));

        // Act: Run the command
        Artisan::call('sync:vehicles');

        Http::assertSent(function ($request) {
            return str_contains($request->url(), $this->prefix . '/authenticate');
        });

        Http::assertSent(function ($request) {
            return str_contains($request->url(), $this->prefix . '/stock') && str_contains($request->url(), '?');
        });

        $this->assertDatabaseHas('vehicles', [
            'auto_trader_id' => 'test2',
            'derivative' => '2.0 test derivative',
            'colour' => 'Blue',
            'attention_grabber' => 'attention test2',
        ]);

        $vehicle = Vehicle::query()->where('auto_trader_id', 'test2')->first();
        $this->assertNotNull($vehicle);

        $this->assertDatabaseHas('vehicle_attribute_values', [
            'owner_id' => $vehicle->id,
            'slug' => 'vat_status',
            'value_text' => 'No VAT',
        ]);

        Event::assertDispatched(NewVehicleImported::class);
    }

    private function mockAutotraderHttpResonses()
    {
        Http::fake([
            $this->prefix . '/authenticate' => Http::response([
                'access_token' => 'fake-token',
                'expires_in' => 3600,
            ], 200),

            $this->prefix . '/stock?*' => Http::response(
                file_get_contents(base_path('tests/data/stock/autotrader-stock-response.json'), true),
                200
            ),
        ]);
    }

    private function mockSettingsFacade()
    {
        Settings::shouldReceive('get')
            ->with('auto-trader-sync-is-published')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('stock-auto-trader-sync-video')
            ->once()
            ->andReturn(true);

        Settings::shouldReceive('get')
            ->with('app-details-country')
            ->twice()
            ->andReturn(null);
    }
}
