<?php

namespace App\Console\Commands;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Mtc\MercuryDataModels\Vehicle;

class RemoveArchivedVehicles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vehicles:purge-soft-deletes';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Remove old/archived vehicles';

    private int $defaultExpiryDays = 90;

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $expiry = Settings::get('app-old-stock-expiry-days') ?: $this->defaultExpiryDays;
        $this->output->info('Removing vehicles that have been removed for X days: ' . $expiry);
        Vehicle::onlyTrashed()
            ->where('deleted_at', '<=', Carbon::now()
            ->subDays($expiry))
            ->chunk(250, fn($collection) => $collection
                ->tap(fn($collection) => $this->output->info('Records to remove in batch: ' . $collection->count()))
                ->each(fn(Vehicle $vehicle) => $this->purge($vehicle)));
    }

    private function purge(Vehicle $vehicle)
    {
        $vehicle->forceDelete();
        $vehicle->specs()->delete();
        $vehicle->mediaUses()->delete();
        $vehicle->features()->delete();
        $vehicle->financeExamples()->delete();
        $vehicle->autoTraderData()->delete();
        $vehicle->conversions()->delete();
        $vehicle->views()->delete();
        $vehicle->attributeValues()->delete();
        $vehicle->labels()->sync([]);
        $vehicle->equipment()->delete();
    }
}
