<?php

namespace App\Crm;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;

class SendCopyViaMail extends \Mtc\Crm\Actions\SendCopyViaMail
{
    use CheckForExpiredAction;

    private string $error;

    private array $recipients = [];

    public function enabled(): bool
    {
        return true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [
            'send_to_field_location' => [
                'type' => 'text',
                'validation' => [
                    'required',
                    'email',
                ],
                'label' => 'Name of question that specifies to where to send submission'
            ],
            'send_to_both' => [
                'type' => 'checkbox',
                'validation' => [
                ],
                'label' => 'Send to both custom email field and action email'
            ],
            'recipient' => [
                'type' => 'text',
                'validation' => [
                    'required',
                    'email',
                ],
                'label' => 'Send Email To'
            ],
            'enquiry_subject' => [
                'type' => 'text',
                'validation' => 'required',
                'label' => 'Enquiry Email Subject',
            ],
            'enquiry_content' => [
                'type' => 'textarea',
                'label' => 'Enquiry Email Message',
            ],
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }
        try {
            $sendToBoth = $action->data['send_to_both'] ?? false;
            $this->action = $action;
            $emails = [];

            $questions_specify_field = $enquiry->form->questions
                ->filter(fn($question) => $question->data['specifies-recipient'] ?? false)
                ->keyBy('id');

            if ($questions_specify_field->isNotEmpty()) {
                $questions_specify_field_ids = $questions_specify_field->pluck('id')->toArray();
                $emails = collect($enquiry->details ?? [])
                    ->filter(fn($answer) => in_array($answer['id'], $questions_specify_field_ids))
                    ->map(function ($answer) use ($questions_specify_field) {
                        $specified_value = collect(explode(
                            "\n",
                            $questions_specify_field[$answer['id']]->data['field-emails-specification'] ?? ''
                        ))
                            ->map(fn($row) => explode(':', $row))
                            ->filter(fn($row) => $row[0] == $answer['answer'])
                            ->map(fn($row) => $row[1])
                            ->first();
                        return collect(explode(',', $specified_value))
                            ->map(fn($email) => trim($email))
                            ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));
                    })
                    ->flatten(1)
                    ->filter()
                    ->toArray();
            }

            if ($this->action->data['send_to_field_location'] ?? null) {
                $specified_field = $this->action->data['send_to_field_location'] ?? null;
                $emails[] = collect($enquiry->details ?? [])
                    ->filter(fn($answer) => $answer['question'] === $specified_field)
                    ->first()['answer'] ?? null;

                $emails = array_filter($emails, fn($email) => $email && filter_var($email, FILTER_VALIDATE_EMAIL));
            }

            if ($sendToBoth || empty($emails)) {
                $action_recipient = $this->action->data['recipient'] ?? null;
                if ($action_recipient) {
                    $raw_emails = explode(',', $action_recipient);
                    foreach ($raw_emails as $email) {
                        if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
                            $emails[] = $email;
                        }
                    }
                }
            }

            $emails = array_unique($emails);
            $this->recipients = $emails;
            if (!empty($emails)) {
                $class = config('crm.new_enquiry_email');
                collect($emails)->each(fn($email) => Mail::to($email)->send(new $class($enquiry)));
                return true;
            }
        } catch (\Exception $exception) {
            $this->error = $exception->getMessage();
            Log::error($exception->getMessage(), [
                'tenant' => tenant('id'),
                'emails' => $emails,
                'exception' => $exception,
            ]);
        }
        return false;
    }

    public function successDetails(): string
    {
        return "Copy sent to " . implode(', ', $this->recipients);
    }

    public function failureReason(): string
    {
        if (empty($this->recipients)) {
            return 'no recipient specified';
        }
        return $this->error ?? 'issue sending mail';
    }
}
