<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Facades\Settings;
use App\Services\DealerWebLmsApi;
use Illuminate\Support\Facades\App;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;

class SendToDealerWebLms implements EnquiryAction, AddsDealershipFields
{
    use CheckForExpiredAction;

    private DealerWebLmsApi $api;

    public function enabled(): bool
    {
        return Settings::get('dealerweblms-crm-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'dealerweb-dealer-id' => [
                'type' => 'text',
                'label' => 'DealerWeb Dealer ID',
            ],
        ];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [
            'dealerweblms-field' => [
                'type' => 'select',
                'label' => 'Field mapped against Dealer Web LMS',
                'choices' => [
                    'username' => 'Username (Third-party unique code)',
                    'dealerNo' => 'Dealer Number (Manufacturer’s dealer number)',
                    'password' => 'Password (16-digit password)',
                    'thirdParty' => 'Third Party Name (Supplier of the lead)',
                    'thirdPartyId' => 'Third Party Unique Customer ID',
                    'version' => 'Version of DealerWeb',
                    'dealerId' => 'Dealer ID (3rd Party Supplier ID)',
                    'leadType' => "Lead type",
                    'title' => 'Title',
                    'forename' => 'Forename',
                    'surname' => 'Surname',
                    'fullName' => 'Full Name (If supplied, split into Forename and Surname)',
                    'salutation' => 'Salutation',
                    'company' => 'Company Name',
                    'houseNameNo' => 'House Name/No.',
                    'street' => 'Street',
                    'locale' => 'Suburb/Locale/District',
                    'city' => 'City',
                    'postcode' => 'Postcode',
                    'telHome' => 'Home Phone Number',
                    'telMobile' => 'Mobile Phone Number',
                    'telWork' => 'Work Phone Number',
                    'email' => 'Email Address',
                    'noMail' => 'Contact allowed via Mail? (0 or 1)',
                    'noSms' => 'Contact allowed via SMS? (0 or 1)',
                    'noEmail' => 'Contact allowed via Email? (0 or 1)',
                    'franchise' => 'Franchise',
                    'model' => 'Model (The model the customer is enquiring about)',
                    'newUsed' => 'New or used',
                    'pexMake' => 'Part Exchange Vehicle Make',
                    'pexModel' => 'Part Exchange Vehicle Model',
                    'pexRegNo' => 'Part Exchange Vehicle Registration Number',
                    'pexMileage' => 'Part Exchange Vehicle Mileage',
                    'detail' => 'Additional Lead Details (Use <![CDATA[ ]]> tag for extra information)',
                    'url' => 'External URL for additional vehicle details (must start with https://)',
                    'enquiryType' => 'Enquiry Type (Refer to Appendix C for valid types)',
                ],
            ]
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }
        $this->api = App::make(DealerWebLmsApi::class);
        return $this->api->sendLead($enquiry, $action);
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return $this->api->getResponseMessage();
    }

    /**
     * Message when failure detected
     *
     * @return string
     */
    public function failureReason(): string
    {
        return $this->api->getResponseMessage() ?? 'Unknown error';
    }
}
