<?php

namespace App\Exports;

use App\Facades\Settings;
use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class AutowebExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::AUTOWEB),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::AUTOWEB)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'feedid' => $this->getValue($vehicle->dealership?->data['autoweb-dealer-id']),
                    'vehicleid' => $this->getValue($vehicle->uuid),
                    'registration' => str_replace(' ', '', $this->getValue($vehicle->registration_number)),
                    'capid' => $this->getValue($vehicle->cap_id),
                    'make' => $this->getValue($vehicle->make?->name),
                    'model' => $this->getValue($vehicle->model?->name),
                    'variant' => $this->getValue($vehicle->derivative),
                    'year' => $this->getValue($vehicle->manufacture_year),
                    'price' => $this->getValue($vehicle->price, true),
                    'mileage' => $this->getValue($vehicle->odometer_mi, true),
                    'colour' => $this->getValue($vehicle->colour),
                    'fueltype' => $this->getValue($vehicle->fuelType?->name),
                    'registrtiondate' => $this->getRegistrationDate($vehicle->first_registration_date),
                    'doors' => $this->getValue($vehicle->door_count),
                    'enginesize' => $this->getValue($vehicle->engine_size_cc),
                    'transmission' => $this->getValue($vehicle->transmission?->name),
                    'description' => $this->getValue($vehicle->description),
                    'options' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),
                    'dealerid' => $vehicle->dealership?->id,
                    'dealername' => $vehicle->dealership?->name,
                    'dealerstreet' => $vehicle->dealership?->address1 . ', ' . $vehicle->dealership?->address1,
                    'dealercounty' => $vehicle->dealership?->county,
                    'dealerpostcode' => '',
                    'dealeremail' => $vehicle->dealership?->email,
                    'picturerefs' => $vehicle->mediaUses
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'Feed_Id',
            'Vehicle_Id',
            'FullRegistration',
            'Cap_Id',
            'Make',
            'Model',
            'Variant',
            'Year',
            'Price',
            'Mileage',
            'Colour',
            'FuelType',
            'Registration_Date',
            'Doors',
            'EngineSize',
            'Transmission',
            'Description',
            'Options',
            'Dealer_Id',
            'Dealer_Name',
            'Dealer_Street_Address',
            'Dealer_County',
            'Dealer_Postcode',
            'Dealer_Crm_Email',
            'PictureRefs',
        ];
    }

    private function getValue($value = false, $forceZeroString = false)
    {
        if (!empty($value)) {
            return $value;
        }

        // item equates to empty
        if ($forceZeroString) {
            return '0';
        }

        return false;
    }

    private function getRegistrationDate($date = false)
    {
        if (empty($date) || $date == '0000-00-00 00:00:00') {
            return false;
        }

        $date_object = \DateTime::createFromFormat('Y-m-d H:i:s', $date);

        if ($date_object) {
            return $date_object->format('Y-m-d');
        }

        return false;
    }
}
