<?php

namespace App\Http\Controllers;

use App\Exports\OfferExport;
use App\Facades\Settings;
use App\Http\Requests\CopyOfferRequest;
use App\Http\Requests\ImportFileRequest;
use App\Http\Requests\SetFeaturedVehicleOffersRequest;
use App\Http\Requests\StoreVehicleOfferRequest;
use App\Http\Requests\UpdateVehicleOfferRequest;
use App\Http\Resources\VehicleOfferListResource;
use App\Http\Resources\VehicleOfferViewResource;
use App\Jobs\BulkImportJob;
use App\OfferRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleOfferContentHistory;
use App\Services\OpenAI;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Mtc\MercuryDataModels\VehicleSpecType;
use Mtc\MercuryDataModels\VehicleStandardEquipment;
use Mtc\MercuryDataModels\VehicleTechnicalData;
use Mtc\VehicleLookup\VehicleLookupService;

class VehicleOfferController extends Controller
{
    public function __construct()
    {
        $this->middleware([
            'permission:edit-offers',
        ]);
    }

    /**
     * Display a listing of the vehicle offers.
     *
     * @param Request $request
     * @return VehicleOfferListResource
     */
    public function index(Request $request): VehicleOfferListResource
    {
        $offers = VehicleOffer::query()
            ->with([
                'make',
                'model',
                'mediaUses.media',
            ])
            ->withViewCount()
            ->withSearchViewCount()
            ->withImageCount()
            ->withEnquiryCount()
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters') ?? '')
            ->when(
                $request->filled('search_term'),
                fn($query) => $query->whereFullText('search_content', $request->input('search_term'))
            )
            ->paginate();
        return new VehicleOfferListResource($offers);
    }

    /**
     * Store a newly created vehicle offer
     *
     * @param \App\Http\Requests\StoreVehicleOfferRequest $request
     * @return VehicleOfferViewResource
     */
    public function store(
        StoreVehicleOfferRequest $request,
        VehicleLookupService $lookupService,
        OfferRepository $repository
    ): VehicleOfferViewResource {
        /** @var VehicleOffer $offer */
        $offer = VehicleOffer::query()->create($request->validated());
        $repository->saveLookupData($offer, $request, $lookupService);
        return new VehicleOfferViewResource($offer->refresh());
    }

    /**
     * Display the specified vehicle offer.
     *
     * @param \Mtc\MercuryDataModels\VehicleOffer $vehicleOffer
     * @return VehicleOfferViewResource
     */
    public function show(VehicleOffer $vehicle_offer): VehicleOfferViewResource
    {
        return new VehicleOfferViewResource($vehicle_offer);
    }

    /**
     * Update the specified vehicle offer
     *
     * @param \App\Http\Requests\UpdateVehicleOfferRequest $request
     * @param \Mtc\MercuryDataModels\VehicleOffer $vehicleOffer
     * @return VehicleOfferViewResource
     */
    public function update(
        UpdateVehicleOfferRequest $request,
        VehicleOffer $vehicleOffer,
        VehicleLookupService $lookupService,
        OfferRepository $repository
    ): VehicleOfferViewResource {
        $repository->save($vehicleOffer, $request, Auth::id());
        $repository->saveLookupData($vehicleOffer, $request, $lookupService);

        return new VehicleOfferViewResource($vehicleOffer);
    }

    /**
     * Remove the specified vehicle offer
     *
     * @param \Mtc\MercuryDataModels\VehicleOffer $vehicleOffer
     * @return bool
     */
    public function destroy(VehicleOffer $vehicleOffer): bool
    {
        return $vehicleOffer->delete();
    }

    /**
     * Show list of featured offers
     *
     * @return VehicleOfferListResource
     */
    public function getFeatured(): VehicleOfferListResource
    {
        return new VehicleOfferListResource(
            VehicleOffer::query()
                ->with([
                    'make',
                    'model',
                    'mediaUses',
                ])
                ->where('featured', 1)
                ->get()
        );
    }

    /**
     * Set offers as featured
     *
     * @param SetFeaturedVehicleOffersRequest $request
     * @return VehicleOfferListResource
     */
    public function setFeatured(SetFeaturedVehicleOffersRequest $request): VehicleOfferListResource
    {
        VehicleOffer::query()
            ->whereNotIn('id', $request->input('offer_ids'))
            ->update([
                'featured' => 0
            ]);

        VehicleOffer::query()
            ->whereIn('id', $request->input('offer_ids'))
            ->update([
                'featured' => 1
            ]);

        return $this->getFeatured();
    }

    public function export(Request $request)
    {
        return Excel::download(new OfferExport(), 'vehicle_offers.xlsx');
    }

    /**
     * Import seo data from file
     *
     * @param ImportFileRequest $request
     * @return bool[]
     */
    public function import(ImportFileRequest $request)
    {
        $tmpName = Str::random() . '.' . $request->file('file')->getClientOriginalExtension();
        $request->file('file')->storeAs('offer-import', $tmpName, ['disk' => 'file-storage']);
        $this->dispatch(new BulkImportJob('offer', 'offer-import/' . $tmpName, 'edit-offers'));
        return [
            'success' => true,
        ];
    }

    public function copy(
        CopyOfferRequest $request,
        VehicleOffer $vehicleOffer,
        OfferRepository $repository
    ): VehicleOfferViewResource {
        $offer = $repository->copy($vehicleOffer, $request);
        return new VehicleOfferViewResource($offer);
    }

    /**
     * Sync offer content with template
     *
     * @param int $offerId
     * @param OfferRepository $repository
     * @return VehicleOfferViewResource|array
     */
    public function syncWithTemplate(int $offerId, OfferRepository $repository): VehicleOfferViewResource|array
    {
        try {
            $repository->syncContentWithTemplate($offerId);
            $offer = VehicleOffer::query()->find($offerId);
        } catch (\Exception $exception) {
            Log::error('Failed to sync offer structure ', [
                'page_id' => $offerId,
                'error' => $exception->getMessage(),
                'tenant' => tenant('id'),
            ]);

            return [
                'success' => false,
                'error' => $exception->getMessage(),
            ];
        }

        return new VehicleOfferViewResource($offer);
    }


    /**
     * @param VehicleOffer $vehicleOffer
     * @param OpenAI $openAI
     * @return JsonResponse
     * @throws ValidationException
     */
    public function getAIDescription(VehicleOffer $vehicleOffer, OpenAI $openAI): JsonResponse
    {
        if (empty(Settings::get('open-ai-api_key'))) {
            throw ValidationException::withMessages([
                'AI Description Not Available'
            ]);
        }

        $description = $openAI->getDescription($vehicleOffer);

        return response()->json([
            'description' => $description
        ]);
    }

    public function restoreVersion(
        VehicleOffer $vehicleOffer,
        VehicleOfferContentHistory $version,
        OfferRepository $repository
    ): VehicleOfferViewResource {
        $repository->restoreVersion($vehicleOffer, $version);
        $vehicleOffer->refresh();
        return new VehicleOfferViewResource($vehicleOffer);
    }
}
