<?php

namespace App\Http\Resources;

use App\Events\LoadDealershipDataFields;
use App\ImportMaps\VehicleImportMap;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;

class ImportMapResource extends JsonResource
{
    public static $wrap = '';
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $this->resource->load('fields');
        $this->resource->row_data = collect($this->resource->data['row_data'] ?? [])->keys();
        $this->resource->data_options = collect($this->getFieldsForType())
                ->map(fn($value, $key) => [
                    'id' => $key,
                    'name' => "$value ($key) ",
                ])
                ->values();
        $resource = $this->resource->toArray();
        $resource['has_header_row'] = $this->resource->data['has_header_row'] ?? true;
        $resource['clear_previous_entries'] = $this->resource->data['clear_previous_entries'] ?? false;
        $resource['fields'] = $this->mapFields();
        $resource['remote_protocols'] = [
            ['value' => 'ftp', 'label' => 'FTP'],
            ['value' => 'sftp', 'label' => 'SFTP'],
        ];
        $resource['schedule_options'] = [
            ['value' => 'daily', 'label' => 'Daily'],
            ['value' => 'twice_daily', 'label' => 'Twice Daily'],
            ['value' => 'every_two_hours', 'label' => 'Every 2 hours'],
            ['value' => 'hourly', 'label' => 'Hourly'],
        ];
        $resource['dealership_fields'] = $this->getDealershipFields();
        $resource['data']['extra_fields'] = $resource['data']['extra_fields'] ?? [];
        return $resource;
    }

    private function getFieldsForType(): array
    {
        return match ($this->resource->type) {
            'vehicle' => (new VehicleImportMap())->columns(),
            default => [],
        };
    }

    private function getDealershipFields()
    {
        return collect(event(new LoadDealershipDataFields()))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->map(function ($field, $key) {
                $field['id'] = $key;
                return $field;
            })
            ->prepend(['id' => 'location_stock', 'label' => 'Stock location'])
            ->values();
    }

    private function mapFields(): Collection
    {
        return $this->resource->fields->keyBy('on_file')
            ->map(function ($field) {
                $field['required_to_import'] = $field['data']['required_to_import'] ?? false;
                return $field;
            });
    }
}
