<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\FormatAsCurrency;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\Vehicle;

class VehicleListResource extends JsonResource
{
    use FormatAsCurrency;

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource
                ->through(fn(Vehicle $vehicle) => $this->mapVehicle($vehicle));
        } else {
            $this->resource = $this->resource
                ->map(fn(Vehicle $vehicle) => $this->mapVehicle($vehicle));
            return parent::toArray($request);
        }

        $resource = $this->resource->toArray();
        $resource['view_columns'] = [
            [
                'id' => 'is_reserved',
                'name' => 'Reserved',
            ],
            [
                'id' => 'is_sold',
                'name' => 'Sold',
            ],
            [
                'id' => 'is_new',
                'name' => 'Is New',
            ],
            [
                'id' => 'registration_number',
                'name' => 'Registration Number',
                'icon' => 'cars',
            ],
            [
                'id' => 'mileage',
                'name' => 'Mileage',
                'icon' => 'gauge',
            ],
            [
                'id' => 'price',
                'name' => 'Price',
                'icon' => 'tag',
            ],
            [
                'id' => 'search_views',
                'name' => 'Search appearances',
                'icon' => 'magnifying-glass',
            ],
            [
                'id' => 'views',
                'name' => 'Ad views',
                'icon' => 'eye',
            ],
            [
                'id' => 'image_count',
                'name' => 'Image count',
                'icon' => 'images',
            ],
            [
                'id' => 'enquiry_count',
                'name' => 'Enquiry count',
                'icon' => 'envelope-dot',
            ],
            [
                'id' => 'days_in_stock',
                'name' => 'Days in stock',
                'icon' => 'truck-clock',
            ],
            [
                'id' => 'days_since_update',
                'name' => 'Days since last update',
                'icon' => 'reply-clock',
            ],
            [
                'id' => 'has_description',
                'name' => 'Has description',
                'icon' => 'text',
            ],
            [
                'id' => 'has_attention_grabber',
                'name' => 'Has attention grabber',
                'icon' => 'text',
            ],
        ];
        $resource['default_columns'] = [
            'registration_number',
            'price',
        ];
        return $resource;
    }

    /**
     * Convert vehicle to the json format used in lists
     *
     * @param Vehicle $vehicle
     * @return array
     */
    protected function mapVehicle(Vehicle $vehicle)
    {
        return [
            'image' => [
                'src' => $vehicle->getPreviewImage('mini-thumb'),
                'alt' => $vehicle->title,
            ],
            'active' => $vehicle->is_published,
            'status' => $vehicle->status,
            'title' => $vehicle->title,
            'description' => $vehicle->derivative,
            'metaItems' => [
                'is_reserved' => $vehicle->is_reserved ? 'Y' : 'N',
                'is_sold' => $vehicle->is_sold ? 'Y' : 'N',
                'is_new' => $vehicle->is_new ? 'Y' : 'N',
                'registration_number' => $vehicle->registration_number,
                'mileage' => $vehicle->odometer_mi,
                'price' => $this->asCurrency($vehicle->price ?? 0, tenant()->currency),
                'search_views' => $vehicle->search_views ?? 0,
                'views' => $vehicle->view_count ?? 0,
                'image_count' => $vehicle->image_count ?? 0,
                'enquiry_count' => $vehicle->enquiry_count ?? 0,
                'days_in_stock' => $vehicle->days_in_stock,
                'days_since_update' => $vehicle->days_since_update,
                'has_description' => !empty($vehicle->description) ? 'Not empty' : 'Empty',
                'has_attention_grabber' => !empty($vehicle->attention_grabber) ? 'Not empty' : 'Empty',
            ],
            'link' => '/manage-content/vehicles/edit/' . $vehicle->id,
            'uuid' => $vehicle->uuid,
            'id' => $vehicle->id,
        ];
    }
}
