<?php

namespace App\Jobs;

use App\Facades\Settings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\SalesChannelHistory;
use Mtc\MercuryDataModels\Vehicle;

class HandleAutoTraderHubNotification implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(private readonly array $input)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        match ($this->input['type'] ?? null) {
            'stock-upsert' => $this->setAutoTraderIdForVehicle(),
            'STOCK_UPDATE' => $this->handleStockUpdate(),
            default => Log::warning('AutoTrader Hub notification - unrecognized action', [
                'input' => $this->input,
                'tenant' => tenant('id'),
            ])
        };
    }

    private function setAutoTraderIdForVehicle(): void
    {
        if ($this->input['result'] === 'success') {
            Vehicle::query()
                ->where('id', $this->input['vehicle_id'])
                ->whereNull('auto_trader_id')
                ->update([
                    'auto_trader_id' => $this->input['stock_id'],
                ]);
        }
        SalesChannelHistory::query()
            ->create([
                'channel' => 'auto-trader',
                'vehicle_id' => $this->input['vehicle_id'],
                'was_successful' => $this->input['result'] === 'success',
                'details' => $this->input['data'],
            ]);
    }

    private function handleStockUpdate(): void
    {
        $statusUpdateData = [];
        $statusUpdateData['is_sold'] = $this->input['data']['metadata']['lifecycleState'] === 'SALE_IN_PROGRESS';
        $statusUpdateData['is_reserved'] = !empty($this->input['data']['adverts']['reservationStatus']);

        if ($this->shouldUnpublish($statusUpdateData)) {
            $statusUpdateData['is_published'] = false;
        }

        if (!empty($statusUpdateData)) {
            Vehicle::query()
                ->where('auto_trader_id', $this->input['stock_id'])
                ->update($statusUpdateData);
        }
    }

    private function shouldUnpublish(array $statusUpdateData): bool
    {
        if (Settings::get('automotive-sales-channels-auto-trader-unpublish-sold') && $statusUpdateData['is_sold']) {
            return true;
        }
        return Settings::get('automotive-sales-channels-auto-trader-unpublish-reserved')
            && $statusUpdateData['is_reserved'];
    }
}
