<?php

namespace App\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Dealership;

class DealershipDetection
{
    /**
     * Determine the DealerID dynamically based on enquiry data.
     *
     * @param string|null $reasonType The reason type (e.g., vehicle, offer).
     * @param array|null $reasonData Data related to the reason (e.g., dealership info).
     * @param string|null $postcode Customer's postcode for proximity matching.
     * @param Collection $dealerships List of dealerships to consider.
     * @param string|null $fallbackDealerId Fallback dealer ID.
     * @return string|null The determined DealerID, or null if not found.
     */
    public function determineDealerId(
        ?string $reasonType,
        ?array $reasonData,
        ?string $postcode,
        Collection $dealerships,
        ?string $fallbackDealerId
    ): ?string {
        // Use dealership data directly from reason if available
        $dealerId = '';
        if (in_array($reasonType, ['vehicle', 'offer'])) {
            $dealerId = $reasonData['dealer-web-lms-dealer-id'] ?? '';
        }

        // Use closest dealership by postcode if no dealer is directly linked
        if (empty($dealerId) && !empty($postcode)) {
            $closestDealer = $this->getClosestDealershipToPostcode($postcode, $dealerships);
            if ($closestDealer) {
                $dealerId = $closestDealer->data['dealer-web-lms-dealer-id'] ?? '';
            }
        }

        // Use fallback dealer ID if no other option is found
        if (empty($dealerId)) {
            $dealerId = $fallbackDealerId;
        }

        return $dealerId;
    }

    /**
     * Get a list of dealerships based on dealer or franchise ID.
     *
     * @param array $actionData Action data containing dealer or franchise IDs.
     * @return Collection A collection of Dealership models.
     */
    public function getDealershipRecipients(array $actionData): Collection
    {
        $dealershipQuery = Dealership::query();

        if (!empty($actionData['dealer-id'])) {
            $dealershipQuery->where('id', '=', $actionData['dealer-id']);
        } elseif (!empty($actionData['franchise-id'])) {
            $dealershipQuery->where('franchise_id', '=', $actionData['franchise-id']);
        }

        return $dealershipQuery->get();
    }

    /**
     * Find the closest dealership to a given postcode.
     *
     * @param string $postcode The postcode to search around.
     * @param Collection $dealerships The collection of dealerships to search within.
     * @return Dealership|null The closest dealership, or null if none found.
     */
    public function getClosestDealershipToPostcode(string $postcode, Collection $dealerships): ?Dealership
    {
        $locatingService = App::make(LocatingService::class);

        try {
            $postcodeLatLong = $locatingService->locate($postcode);
        } catch (\Exception $exception) {
            Log::info('Postcode not found: ' . $postcode);
            return null;
        }

        if ($this->postcodeNotFound($postcodeLatLong)) {
            return null;
        }

        $closestDealerId = $dealerships
            ->mapWithKeys(fn($dealer) => [
                $dealer->id => explode(',', $dealer->coordinates),
            ])
            ->filter(fn($coords) => count($coords) === 2)
            ->map(fn($coords) => $locatingService->getDistanceBetween(
                $postcodeLatLong->lat(),
                $postcodeLatLong->lng(),
                $coords[0],
                $coords[1],
            ))
            ->sort()
            ->keys()
            ->first();

        return Dealership::query()->find($closestDealerId);
    }

    /**
     * Determine if a postcode lookup failed.
     *
     * @param $postcodeLatLong The latitude/longitude result from a postcode lookup.
     * @return bool True if the postcode was not found, false otherwise.
     */
    private function postcodeNotFound($postcodeLatLong): bool
    {
        return (
            empty($postcodeLatLong)
            || (
                empty($postcodeLatLong->lat())
                && empty($postcodeLatLong->lng())
            )
        );
    }
}
