<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class SiteRepository
{
    public function vehicleUrl(Vehicle $vehicle, bool $withAccess = false): ?string
    {
        try {
            return $this->baseUrl() . $vehicle->urlPath() . ($withAccess ? $this->accessParam($vehicle) : '');
        } catch (ModelNotFoundException $exception) {
            return null;
        }
    }

    public function pageUrl(Page $page, bool $withAccess = false): ?string
    {
        try {
            $url_path = $page->urlPath();
            if ($url_path === '/home') {
                $url_path = '/';
            }
            return $this->baseUrl() . $url_path . ($withAccess ? $this->accessParam($page) : '');
        } catch (ModelNotFoundException $exception) {
            return null;
        }
    }

    public function offerUrl(VehicleOffer $offer, bool $withAccess = false): ?string
    {
        try {
            return $this->baseUrl() . $offer->urlPath() . ($withAccess ? $this->accessParam($offer) : '');
        } catch (ModelNotFoundException $exception) {
            return null;
        }
    }

    public function newCarUrl(NewCar $newCar, bool $withAccess = false): ?string
    {
        try {
            return $this->baseUrl() . $newCar->urlPath() . ($withAccess ? $this->accessParam($newCar) : '');
        } catch (ModelNotFoundException $exception) {
            return null;
        }
    }

    public function url(string $endpoint = ''): ?string
    {
        try {
            return $this->baseUrl() . '/' . ltrim($endpoint, '/');
        } catch (ModelNotFoundException $exception) {
            return null;
        }
    }

    protected function baseUrl(): string
    {
        $domain = tenant()->domains()
            ->where('verified', 1)
            ->orderByDesc('primary')
            ->firstOrFail();

        $protocol = config('app.env') === 'local' ? 'http://' : 'https://';

        return $protocol . $domain->domain;
    }

    private function accessParam(Model $model): string
    {
        return  '?' . http_build_query([
            'a' => base64_encode($model->id . '-' . $model->slug),
        ]);
    }
}
