<?php

namespace Tests\Unit;

use App\Facades\Settings;
use Database\Seeders\Tenant\AutomotiveSettingSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Models\Media as MediaModel;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\TestCase;

class VehicleTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    /**
     * A basic test example.
     *
     * @return void
     */
    public function testVehicleHasImages()
    {
        $this->seed(AutomotiveSettingSeeder::class);

        $vehicle = Vehicle::factory()->create();
        $media = MediaModel::factory(3)->create();

        Media::setUsesForModel($media->pluck('id')->toArray(), $vehicle);

        $this->assertEquals(3, $vehicle->mediaUses()->count());
    }


    public function testVehicleSlug()
    {
        $this->seed(AutomotiveSettingSeeder::class);
        Settings::update('automotive-vehicles-url_building_format', 'vrm-make-model');
        $make = VehicleMake::factory()->create(['name' => 'BMW']);
        $model = VehicleModel::factory()->create(['name' => 'i3']);
        $vehicle = Vehicle::factory()->make([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'registration_number' => 'foo',
        ]);
        $vehicle->slug = null;
        $vehicle->save();

        $this->assertEquals('foo-bmw-i3', $vehicle->slug);

        Settings::update('automotive-vehicles-url_building_format', 'title');
        $vehicle2 = Vehicle::factory()->make(['title' => 'Hello my Car']);
        $vehicle2->slug = null;
        $vehicle2->save();
        $this->assertEquals('hello-my-car', $vehicle2->slug);

        Settings::update('automotive-vehicles-url_building_format', 'uuid');
        $vehicle3 = Vehicle::factory()->make(['uuid' => 'baz']);
        $vehicle3->slug = null;
        $vehicle3->save();
        $this->assertEquals('baz', $vehicle3->slug);
    }

    public function testSearchContentSetWhenSaving()
    {
        $make = VehicleMake::factory()->create(['name' => 'bmw']);
        $model = VehicleModel::factory()->create(['name' => '330i']);
        $fuelType = FuelType::factory()->create(['name' => 'Petrol']);
        $bodyStyle = BodyStyleType::factory()->create(['name' => 'Coupe']);
        $vehicle = Vehicle::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'fuel_type_id' => $fuelType->id,
            'body_style_id' => $bodyStyle->id,
            'derivative' => 'e92 Petrol Coupe',
            'registration_number' => 'AC69 DDE',
            'description' => '<p> hello world</p>',
            'colour' => 'black',
            'manufacture_year' => 2007,
        ]);

        $this->assertStringContainsString('bmw', $vehicle->search_content);
        $this->assertStringContainsString('330i', $vehicle->search_content);
        $this->assertStringContainsString('petrol', $vehicle->search_content);
        $this->assertStringContainsString('e92', $vehicle->search_content);
        $this->assertStringContainsString('ac69', $vehicle->search_content);
    }
}
