<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\Crm\Contracts\CustomerNote;
use Mtc\MercuryDataModels\Enquiry;
use Mtc\MercuryDataModels\Traits\FormatAsCurrency;

class CustomerView extends JsonResource
{
    use FormatAsCurrency;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param Request $request
     * @return array
     */
    public function toArray($request)
    {
        $this->resource->total_enquiries = $this->resource->enquiries()->count();
        $this->resource->total_reservations = $this->resource->reservations()->count();
        $this->resource->total_bookings = $this->resource->bookings()->count();

        $this->resource->notes = $this->resource->notes()
            ->with('author')
            ->latest()
            ->get()
            ->map(fn($note) => $this->mapNote($note));
        $this->resource->enquiries = $this->resource->enquiries()
            ->with('type')
            ->latest()
            ->take(5)
            ->get()
            ->map(fn($enquiry) => $this->mapEnquiry($enquiry));
        $this->resource->bookings = $this->resource->bookings()->latest()->take(5)->get()
            ->map(fn($booking) => $this->mapBooking($booking));
        $this->resource->reservations = $this->resource->reservations()
            ->with('vehicle')
            ->latest()
            ->take(5)
            ->get()
            ->map(fn($reservation) => $this->mapReservation($reservation));
        return parent::toArray($request);
    }

    private function mapNote(CustomerNote $note): array
    {
        return [
            'author' => $note->author?->name,
            'type' => $note->type,
            'severity' => $note->severity,
            'when' => $note->created_at->format('d/m/Y H:i'),
            'details' => $note->details,
        ];
    }

    private function mapEnquiry(Enquiry $enquiry): array
    {
        return [
            'when' => $enquiry->created_at->format('d/m/Y H:i'),
            'type' => $enquiry->type?->name,
            'id' => $enquiry->id,
        ];
    }

    private function mapBooking($booking): array
    {
        return [
            'when' => $booking->booking_time->format('d/m/Y H:i'),
            'vehicle' => $booking->vehicle_data
                ? collect($booking->vehicle_data)->filter()->implode(' ')
                : $booking->registration_number,
            'details' => collect([
                $booking?->location?->name,
                $booking?->packages?->pluck('name')->implode(','),
                $booking?->deliveryOption?->name,
                ])->filter()->implode(' '),
            'id' => $booking->id,

        ];
    }

    private function mapReservation($reservation): array
    {
        return [
            'when' => $reservation->created_at->format('d/m/Y H:i'),
            'details' => $reservation->data['vehicle']['description'] ?? collect([
                    $reservation->vehicle?->make?->name,
                    $reservation->vehicle?->model?->name,
                    $reservation->vehicle?->derivative,
                    $reservation->vehicle?->registration_number,
                    $this->asCurrency($reservation->vehicle?->price),
                ])->filter()->implode(' '),
            'id' => $reservation->id,
        ];
    }
}
