<?php

namespace App\Http\Controllers;

use App\Facades\Settings;
use App\Http\Requests\ViewVehicleRequest;
use App\Http\Resources\VehicleQuickViewResource;
use App\Http\Resources\VehicleResource;
use App\VehicleRepository;
use Mtc\MercuryDataModels\Vehicle;
use App\Repositories\RecentlyViewed;
use Illuminate\Http\Request;

class VehicleController extends Controller
{
    /**
     * Show vehicle FPA details
     *
     * @param Vehicle $vehicle
     * @return array
     */
    public function show(
        ViewVehicleRequest $request,
        Vehicle $vehicle,
        RecentlyViewed $recentlyViewed,
        VehicleRepository $repository
    ) {
        $repository->trackView($vehicle);
        $recentlyViewed->add('vehicles', $vehicle->id);
        $crossSell = [];
        if (Settings::get('vehicles-cross-sell-enabled')) {
            $crossSell = [
                'name' => __('automotive.cross-sell.similar'),
                'items' => (new VehicleRepository())->similar($vehicle),
            ];
        }
        return [
            'data' => new VehicleResource($vehicle),
            'cross_sell' => $crossSell,
        ];
    }

    public function quickView(
        ViewVehicleRequest $request,
        Vehicle $vehicle,
        VehicleRepository $repository
    ) {
        $repository->trackQuickView($vehicle);
        return [
            'data' => new VehicleQuickViewResource($vehicle),
        ];
    }

    /**
     * Get list of vehicles for compare
     *
     * @param Request $request
     * @param VehicleRepository $repository
     * @return array
     */
    public function compare(Request $request, VehicleRepository $repository): array
    {
        $vehicleSlugs = $request->input('slugs') ?? [];

        return [
            'data' => $repository->compare($vehicleSlugs),
        ];
    }

    /**
     * Get list of recently viewed vehicles
     *
     * @param Vehicle $vehicle
     * @param Request $request
     * @return array
     */
    public function recentlyViewed(Vehicle $vehicle, Request $request)
    {
        $vehicleSlugs = $request->input('slugs') ?? [];

        return [
            'recently_viewed' => [
                'name' => __('automotive.recently_viewed'),
                'items' => (new VehicleRepository())->recentlyViewed($vehicle, $vehicleSlugs),
            ]
        ];
    }

    /**
     * Get list of recently added vehicles
     *
     * @param Request $request
     * @return array
     */
    public function recentlyAdded(Request $request): array
    {
        $limit = $request->input('limit') ?? 4;

        return [
            'recently_added' => [
                'name' => __('automotive.recently_added'),
                'items' => (new VehicleRepository())->recentlyAdded($limit, [
                    'is_new' => false
                ]),
            ]
        ];
    }

    /**
     * Get list of featured vehicles
     *
     * @param Request $request
     * @param Vehicle $vehicle
     * @param VehicleRepository $repository
     * @return array
     */
    public function featured(Request $request, Vehicle $vehicle, VehicleRepository $repository): array
    {
        return [
            'name' => __('automotive.cross-sell.featured'),
            'items' => $repository
                ->featured(!empty($vehicle->id) ? $vehicle : null, 4, $request->input())
                ->toArray($request),
        ];
    }
}
