<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\HasPlaceholderImage;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\SeoDefault;

class DealershipView extends JsonResource
{
    use HasPlaceholderImage;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->load([
            'mediaUses.media',
        ]);

        return [
            'name' => $this->resource->name,
            'slug' => $this->resource->slug,
            'contact_no' => $this->resource->contact_no,
            'email' => $this->resource->email,
            'coordinates' => $this->resource->coordinates,
            'open_times' => $this->resource->open_times,
            'is_open' => $this->resource->is_open,
            'opens_at' => $this->resource->opens_at,
            'closes_at' => $this->resource->closes_at,
            'alt_open_times' => Settings::get('dealership-second-open-times')
                ? $this->resource->alt_open_times
                : [],
            'alt_open_time_name' => Settings::get('dealership-secondary-open-close-time-name'),
            'notes' => $this->resource->notes,
            'seo' => $this->fillSeo(),
            'address' => collect([
                $this->resource->address1,
                $this->resource->address2,
                $this->resource->city,
                $this->resource->county,
                $this->resource->postcode,
            ])->filter()->implode(', '),
            'images' => $this->getGallery($this->resource)->map(fn($image) => [
                'src' => $image['src_large'],
                'alt' => $image['alt_text'],
                'sizes' => $image['sizes'] ?? [],
            ]),
            'franchise' => $this->resource->franchise,
        ];
    }

    private function fillSeo(): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $default = SeoDefault::query()->where('section', 'dealership')->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }

    protected function largeImageDimensions(): string
    {
        return 'full-width-banner';
    }

    protected function mediumImageDimensions(): string
    {
        return '300x300';
    }

    protected function thumbnailImageDimensions(): string
    {
        return '300x300';
    }
}
