<?php

namespace App;

use App\Http\Requests\PartialEnquirySubmissionRequest;
use App\Http\Requests\SubmitEnquiryRequest;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\Scopes\OnlySubmitted;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class EnquiryFormHandler
{
    /**
     * @param SubmitEnquiryRequest $request
     * @return Enquiry
     */
    public function partialSubmit(PartialEnquirySubmissionRequest $request): Model
    {
        return Enquiry::query()
            ->withoutGlobalScope(OnlySubmitted::class)
            ->updateOrCreate([
                'id' => $request->input('enquiry_id'),
            ], [
                'reason_id' => $this->getReasonId($request),
                'reason_type' => $this->getReasonType($request),
                'form_id' => $request->input('id'),
                'email' => $this->getEmailFromRequest($request),
                'type_id' => $request->form()->type_id,
                'source' => $request->input('source'),
                'status_id' => 0,
                'title' => $this->getTitleFromRequest($request),
                'message' => '',
                'details' => $this->details($request),
                'valuation_id' => $request->input('valuation_id'),
                'data' => [
                    'questions' => $request->form()?->questions()->pluck('data', 'id'),
                ],
                'assigned_user_id' => $request->form()->initial_assignee,
            ]);
    }

    /**
     * Handle incoming Request
     *
     * @param SubmitEnquiryRequest $request
     * @return array|string[]
     */
    public function handle(SubmitEnquiryRequest $request): array
    {
        if ($this->processRequest($request)) {
            return [
                'status' => 'ok',
                'message' => $this->getSuccessMessage($request),

            ];
        }
        return [
            'status' => 'error',
            'message' => $this->getErrorMessage()
        ];
    }

    /**
     * Process enquiry request to record
     * @param SubmitEnquiryRequest $request
     * @return bool
     */
    protected function processRequest(SubmitEnquiryRequest $request): bool
    {
        try {
            Enquiry::query()
                ->withoutGlobalScope(OnlySubmitted::class)
                ->updateOrCreate([
                    'id' => $request->input('enquiry_id'),
                ], [
                    'reason_id' => $this->getReasonId($request),
                    'reason_type' => $this->getReasonType($request),
                    'form_id' => $request->input('id'),
                    'source' => $request->input('source'),
                    'email' => $this->getEmailFromRequest($request),
                    'type_id' => $request->form()->type_id,
                    'status_id' => $request->form()->initial_status_id,
                    'title' => $this->getTitleFromRequest($request),
                    'message' => '',
                    'details' => $this->details($request),
                    'valuation_id' => $request->input('valuation_id'),
                    'data' => [
                        'questions' => $request->form()?->questions()->pluck('data', 'id'),
                    ],
                    'assigned_user_id' => $request->form()->initial_assignee,
                ]);
        } catch (\Exception $exception) {
            Log::error('Failed to process Enquiry Submission', [
                'input' => $request->input(),
                'error' => $exception->getMessage(),
            ]);
            return false;
        }
        return true;
    }

    protected function details(SubmitEnquiryRequest $request)
    {
        return collect($request->input('questions'))
            ->map(fn($answer, $id) => [
                'id' => $id,
                'answer' => $answer,
                'question' => $request->form()->questions->where('id', $id)->first()->name ?? '',
            ])->toArray();
    }

    /**
     * Success message
     *
     * @param SubmitEnquiryRequest $request
     * @return string
     */
    protected function getSuccessMessage(SubmitEnquiryRequest $request): string
    {
        return $request->form()->success_message ?? '';
    }

    /**
     * Error message
     *
     * @return string
     */
    protected function getErrorMessage(): string
    {
        return 'Failed to submit your enquiry. Please make sure all required fields are filled and try again';
    }

    /**
     * Get the enquiry title/subject from form fields and request data
     *
     * @param SubmitEnquiryRequest $request
     * @return string
     */
    protected function getTitleFromRequest(SubmitEnquiryRequest $request): string
    {
        return $request->form()
            ->questions
            ->filter(fn($question) => $question->is_enquiry_subject)
            ->sortBy(fn($question) => $question->order)
            ->map(function ($question) use ($request) {
                $inputValue = $request->input('questions.' . $question->id);

                if ($this->isRelatedObject($question->type)) {
                    return $this->getRelatedObject($question->type, (int)$inputValue);
                }

                if (is_array($inputValue)) {
                    return implode(', ', $inputValue);
                }

                return $inputValue;
            })
            ->implode(' ');
    }

    /**
     * Find email field value based off the form fields and request
     *
     * @param SubmitEnquiryRequest $request
     * @return string|null
     */
    protected function getEmailFromRequest(SubmitEnquiryRequest $request): string|null
    {
        $question_id = $request->form()
            ->questions
            ->filter(fn($question) => $question->is_customers_email)
            ->first()
            ->id ?? null;

        return $question_id ? $request->input('questions.' . $question_id) : '';
    }

    /**
     * Check if the type is a related object
     *
     * @param string $type
     * @return bool
     */
    protected function isRelatedObject(string $type)
    {
        return in_array($type, [
            'page_id',
            'vehicle_id',
            'offer_id',
            'dealership_id',
        ]);
    }

    protected function getRelatedObject(string $type, int $id)
    {
        return match ($type) {
            'page_id' => Page::query()->find($id)->title ?? '',
            'vehicle_id' => Vehicle::query()->find($id)->title ?? '',
            'offer_id' => VehicleOffer::query()->find($id)->name ?? '',
            'dealership_id' => Dealership::query()->find($id)->name ?? '',
        };
    }

    /**
     *
     * @param Request $request
     * @return string|null
     */
    protected function getReasonType(Request $request): ?string
    {
        return $request->form()
            ->questions
            ->filter(fn($question) => $this->isRelatedObject($question->type))
            ->map(fn($question) => str_replace('_id', '', $question->type))
            ->first();
    }

    /**
     * Get the reason ID for the enquiry
     *
     * @param Request $request
     * @return mixed
     */
    protected function getReasonId(Request $request)
    {
        return $request->form()
            ->questions
            ->filter(fn($question) => $this->isRelatedObject($question->type))
            ->map(fn($question) => $request->input('questions.' . $question->id))
            ->first();
    }
}
