<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\RetrievesContent;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\MercuryDataModels\SeoData;
use Mtc\MercuryDataModels\SeoDefault;

class PageResource extends JsonResource
{
    use FranchiseData;
    use RetrievesContent;

    protected Collection $mediaSizes;

    private const UI_RESOURCE_PREFIX = 'App\\Http\\Resources\\';

    private Request $request;

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->request = $request;
        if ($request->filled('v')) {
            $this->resource->data = $this->previewVersion($request->input('v'));
        }
        if (empty($this->resource->data)) {
            $this->resource->load([
                'rootLevelContent.mediaUses.media',
                'rootLevelContent.subContent.mediaUses.media',
                'rootLevelContent.subContent.subContent.mediaUses.media',
                'rootLevelContent.subContent.subContent.subContent.mediaUses.media',
                'rootLevelContent.subContent.subContent.subContent.subContent.mediaUses.media',
                'rootLevelContent.contentElement',
                'rootLevelContent.globalContent.contentElement',
                'rootLevelContent.subContent.contentElement',
                'rootLevelContent.subContent.globalContent.contentElement',
                'rootLevelContent.subContent.subContent.contentElement',
                'rootLevelContent.subContent.subContent.globalContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.subContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.globalContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.subContent.globalContent.contentElement',
                'categories' => fn($query) => $query->orderBy('name'),
                'tags' => fn($query) => $query->orderBy('name'),
            ]);
        }

        $this->resource->setHidden([
            'text',
        ]);

        $resource = $this->resource->toArray();
        $resource['content'] = $this->getContent($this->resource->rootLevelContent);
        $resource['isFranchise'] = $this->isFranchise();
        $resource['franchise'] = $this->franchiseData();
        $resource['seo'] = $this->fillSeo();
        $resource['published_at'] = $this->resource->published_at?->format('F j, Y');
        $resource['categories'] = $this->resource->categories->pluck('name', 'slug')->toArray();
        $resource['tags'] = $this->resource->tags->pluck('name', 'slug')->toArray();

        if ($this->resource->template?->slug) {
            $resource['template_slug'] = $this->resource->template->slug;
        }

        unset($resource['password']);
        unset($resource['root_level_content']);
        return $resource;
    }

    private function previewVersion(mixed $versionId): ?Collection
    {
        /** @var VersionModel $version */
        $version = $this->resource->versions()
            ->with([
                'content.mediaUses.media',
                'content.subContent.mediaUses.media',
                'content.subContent.subContent.mediaUses.media',
                'content.subContent.subContent.subContent.mediaUses.media',
                'content.contentElement',
                'content.globalContent.contentElement',
                'content.subContent.contentElement',
                'content.subContent.globalContent.contentElement',
                'content.subContent.subContent.contentElement',
                'content.subContent.subContent.globalContent.contentElement',
                'content.subContent.subContent.subContent.contentElement',
                'content.subContent.subContent.subContent.globalContent.contentElement',
            ])
            ->where('uuid', $versionId)
            ->first();

        return $version ? $this->mapContentList($version->content) : null;
    }

    private function fillSeo(): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $data = SeoData::query()->where('path', $this->request->header('x-path'))->first();
        if ($data) {
            return $data->toArray();
        }

        $default = SeoDefault::query()->where('section', 'page')->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{CONTENT_EXCERPT}}' => $this->resource->excerpt,

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }
}
